import numpy as np
from scipy.optimize import minimize
#from statsmodels.tsa.arima_model import ARMA
from scipy.stats import norm

#subroutines for time checking
import time
def totime(t):
    return time.ctime(t+15*3600)[11:-5]
###############################


def ARV(X, K, I=1):
    ## Computing averaged Sparse RV with tuning parameter K
    # =========================================================================
    # INPUT  - X:     log(price) series, NOT the raw price series
    #        - K:     tuning parameter
    #        - I:     indicator:
    #                 I = 1: default level, compute K sparse realized variances and take the average
    #                 I = 0: only compute the first sparse realized variance
    # OUTPUT - RVavg: realized variance

    K_float = np.floor(K)  # robustness
    n = X.size

    if I == 1:
        RVs = np.zeros(K)

        for k in range(0, K):
            loc = range(k, (k+K*int(np.floor((n-1-k)/K_float)))+1, K)
            RVs[k] = sum(np.diff(X[loc])**2)

        RVavg = np.mean(RVs)
    else:
        loc = range(0, (1+K*int(np.floor((n-2)/K_float)))+1, K)
        RVavg = sum(np.diff(X[loc])**2)

    return RVavg

################################   QMLE related functions

def MA1_triang_loglik(param, y):
    # simpler form of the MA(1) log-lik based on triangularization of the Cov
    # for ease of the optimization procedure

    n = y.size
    eta = param[0]
    g2 = param[1]

    if (eta<1) and (eta>-1) and (g2>0): #put (eta<0) if noise not close to 0, we extended it as in Ait-Sahalia and Xiu (2017)
        etavec = eta**(2*np.array(range(n+1)))
        etasum = np.cumsum(etavec)
        etasumi = etasum[1:]  # n-dim vector where position i has 1 + eta^2 + .... + eta^(2(i+1))
        etasumim1 = etasum[:-1]  # n-dim vector where position i has 1 + eta^2 + .... + eta^(2i)
        dvec = g2*etasumi/etasumim1

        ytilde = np.zeros(n)
        ytilde[0] = y[0]

        for i in range(1, n):
            ytilde[i] = y[i]-ytilde[i-1]*eta*etasumim1[i-1]/etasumim1[i]

        temp = - sum(np.log(dvec))/2-sum((ytilde**2)/dvec)/2

        loglik = - temp  # don't forget the minus to get a max!
    else:
        loglik = 10**25

    return loglik

def spread_loglik(param, y, info, delta):

    n = y.size
    sigma2 = param[0]
    theta = param[1]

    if (sigma2 > 0):
        loglik = - n*np.log(sigma2*delta)/2 - sum((y - info*theta)**2)/(sigma2*delta)
    else:
        loglik = 10**25

    return loglik

def toEtaGamma(sigma2, a2, delta):
    return (-1-sigma2*delta/(2.0*a2)+1.0/(2.0*a2)*np.sqrt(sigma2*delta*(4*a2+sigma2*delta)),
            a2+sigma2*delta/2.0+0.5*np.sqrt(sigma2*delta*(4*a2+sigma2*delta)))

def fromEtaGamma(eta, gamma2, delta):
    return (gamma2/delta*(1+eta)**2, -gamma2*eta)

def QMLE_manual(Y, D, returnVal = False):
    ## Compute the QMLE estimator when the microstructure produces white noises
    # =========================================================================
    # INPUT  - Y:      log(price) series, NOT raw prices
    #          D:      length of the sampling interval
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta, np.diff(Y)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x
    if returnVal:
        def l(theta): return MA1_triang_loglik(theta, np.diff(Y))
        loglik = l(param)
    # param2 = minimize(lambda theta : MA1_triang_loglik(theta,np.diff(Y)), parastart, method='Powell', options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    if returnVal:
        return [IVmle, a2mle, etamle, g2mle, loglik]
    else:
        return [IVmle, a2mle, etamle, g2mle]

def QMLE_manual_roll(Y, info, D, returnVal = False):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 9)
    knownNoisestart = knownNoise_Yingying_roll(Y, info, D)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - theta[2]*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x
    # param2 = minimize(lambda theta : MA1_triang_loglik(theta,np.diff(Y)), parastart, method='Powell', options={'xtol': 1e-9, 'disp': False}).x
    if returnVal:
        def l(theta): return MA1_triang_loglik(theta[0:2], np.diff(Y-theta[2]*info))

        loglik = -l(param)
    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    if returnVal:
        return [IVmle, a2mle, knownNoisemle, etamle, g2mle, loglik ]
    else:
        return [IVmle, a2mle, knownNoisemle, etamle, g2mle]

def QMLE_manual_glostenharris(Y, info, volume, D):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 9) #sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart1 = knownNoise_Yingying_roll(Y, info, D)
    knownNoisestart2 = 8 * (10 **(-10))

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart1, knownNoisestart2]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] + theta[3]*volume)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x
    # param2 = minimize(lambda theta : MA1_triang_loglik(theta,np.diff(Y)), parastart, method='Powell', options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    knownNoisemle2 = param[3]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, etamle, g2mle]

def QMLE_manual_glostenharrisNoRoll(Y, info, volume, D):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = np.sqrt(a2start)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2]*volume)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x
    # param2 = minimize(lambda theta : MA1_triang_loglik(theta,np.diff(Y)), parastart, method='Powell', options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_glostenharrists(Y, info, volume, ts, D):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = np.sqrt(a2start)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart, 0, 0]

    ## time set up
    #ts2 = np.zeros(ts.size)
    #tsdif = np.diff(ts)
    #ts2[0] = min(1/ts[0],20)
    #for i in range(ts2.size-1) :
    #    ts2[i+1] = min(1/tsdif[i], 20)
    ts2 = np.ones(ts.size)
    ts2[1:] =1/(1+ np.diff(ts))
    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] + theta[3] * volume + theta[4] * ts2) * info)),
                     parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    knownNoisemle2 = param[3]
    knownNoisemle3 = param[4]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, knownNoisemle3, etamle, g2mle]

def QMLE_manual_ts(Y, info, ts, D):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 9)
    knownNoisestart = 5 * (10 ** (- 6))

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    ## time set up
    #ts2 = np.zeros(ts.size)
    #tsdif = np.diff(ts)
    #ts2[0] = min(1/ts[0],20)
    #for i in range(ts2.size-1) :
    #    ts2[i+1] = min(1/tsdif[i], 20)
    ts2 = np.ones(ts.size)
    ts2[1:] =1/(1+ np.diff(ts))

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] * ts2) * info)),
                     parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_spread(Y, info, spread, D, returnVal = False):
    ## Compute the QMLE estimator when the microstructure follows spread model
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 10)
    knownNoisestart = knownNoise_Yingying_spread(Y, info, spread, D)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2]*spread)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    if returnVal:
        def l(theta): return MA1_triang_loglik(theta[0:2], np.diff(Y-theta[2]*spread*info))

        loglik = -l(param)
        newParam = fromEtaGamma(param[0], param[1], deltabar)
        #print(MA1_triang_loglik(param[0:2], np.diff(Y - (param[2]*spread)*info)))
        #print(spread_loglik([newParam[0],param[2]], np.diff(Y), np.diff(spread*info), deltabar))
    #print(MA1_triang_loglik(param[0:2], np.diff(Y - (param[2]*spread)*info)))
    #print(spread_loglik(param[0:2], np.diff(Y - (param[2] * spread) * info), np.diff(info), deltabar))
    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    if returnVal:
        return [IVmle, a2mle, knownNoisemle1, etamle, g2mle, loglik]
    else:
        return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_spreadLog(Y, info, spread, D):
    ## Compute the QMLE estimator when the microstructure follows spread model
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter
    #          actually calculates exp signed spread

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 10)
    knownNoisestart = knownNoise_Yingying_spread(Y, info, spread, D)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - ((theta[2]*spread)/(1+theta[2]*spread))*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_spreadRoll(Y, info, spread, D,returnVal=False):
    ## Compute the QMLE estimator when the microstructure follows spread + Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = np.sqrt(a2start)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart, 0]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] + theta[3]*spread)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x
    #print(MA1_triang_loglik(param[0:2], np.diff(Y - (param[2] + param[3]*spread)*info)))

    if returnVal:
        def l(theta): return MA1_triang_loglik(theta[0:2], np.diff(Y-(theta[2] + theta[3]*spread)*info))

        loglik = -l(param)
    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    knownNoisemle2 = param[3]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    if returnVal:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, etamle, g2mle,loglik]
    else:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, etamle, g2mle]

def QMLE_manual_spreadRollTs(Y, info, spread, ts, D,returnVal=False):
    deltabar = float(D) / (Y.size - 1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size))) / D
    # full sampling for noise
    a2start = 10 ** (- 10) #sum(np.diff(Y) ** 2) / (2 * (Y.size - 1))
    knownNoisestart1 = -3 * (10 ** (- 5))
    knownNoisestart2 = 1
    knownNoisestart3 = -5 * 10 ** (- 7)


    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart1, knownNoisestart2, knownNoisestart3]

    ## time set up
    #ts2 = np.zeros(ts.size)
    #tsdif = np.diff(ts)
    #ts2[0] = min(1 / ts[0], 20)
    #for i in range(ts2.size - 1):
    #    ts2[i + 1] = min(1 / tsdif[i], 20)
    #print(ts)
    #print(spread)
    #print(info)
    ts2 = np.ones(ts.size)
    ts2[1:] =1/(1+ np.diff(ts))
    #print(ts2)
    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] + theta[3] * spread +  theta[4] * ts2) * info)),
                     parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    if returnVal:
        def l(theta): return MA1_triang_loglik(theta[0:2], np.diff(Y-(theta[2] + theta[3]*spread+ theta[4] * ts2)*info))

        loglik = -l(param)
    #print(MA1_triang_loglik(param[0:2], np.diff(Y - (param[2] + param[3] * spread + param[4] * volume + param[5] * volLim + param[6] * ts2) * info)))
    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1 = param[2]
    knownNoisemle2 = param[3]
    knownNoisemle3 = param[4]

    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D * volmle

    if returnVal:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, knownNoisemle3, etamle, g2mle,loglik]
    else:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, knownNoisemle3, etamle,
                g2mle]

def QMLE_manual_spreadLag(Y, info, spread, D):
    ## Compute the QMLE estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = np.sqrt(a2start)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, 0, 0]
    spreadlag = np.zeros(Y.size)
    spreadlag[0] = spread[0]
    for i in range(Y.size - 1):
        spreadlag[i+1] = spread[i]

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:3], np.diff(Y - (theta[2]*spread + theta[3]*spreadlag)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    knownNoisemle2 = param[3]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, etamle, g2mle]

def QMLE_manual_volLim(Y, info, volLim, D):
    ## Compute the QMLE estimator when the microstructure follows spread model
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = 10 ** (- 9)#sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = 2 * (10 ** (- 7))

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2]*volLim)*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_OPI(Y, OPI, D):
    ## Compute the QMLE estimator when the microstructure follows spread model
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart = np.sqrt(a2start)

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2]*OPI))), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, etamle, g2mle]

def QMLE_manual_spreadVolLim(Y, info, spread, volLim, D):
    ## Compute the QMLE estimator when the microstructure follows spread model
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - Volmle: MLE estimate for integrated volatility
    #          a2mle:  MLE estimate for noise variance
    #          knownNoisemle:  MLE estimate for the known noise parameter

    deltabar = float(D)/(Y.size-1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size)))/D
    # full sampling for noise
    a2start = sum(np.diff(Y)**2)/(2*(Y.size-1))
    knownNoisestart1 = np.sqrt(a2start)
    knownNoisestart2 = 0

    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart1, knownNoisestart2]

    # print(parastart)

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:3], np.diff(Y - (theta[2]*volLim + (theta[3]*spread))*info)), parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1=param[2]
    knownNoisemle2 = param[3]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D*volmle

    return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, etamle, g2mle]

def QMLE_manual_spreadRollVolTsVollim(Y, info, spread, volume, volLim, ts, D,returnVal=False):
    deltabar = float(D) / (Y.size - 1)
    # starting points of optimization (search for MLE)
    # sparse sampling for volatility
    vol2start = ARV(Y, int(np.sqrt(Y.size))) / D
    # full sampling for noise
    a2start = 10 ** (- 10) #sum(np.diff(Y) ** 2) / (2 * (Y.size - 1))
    knownNoisestart1 = -3 * (10 ** (- 5))
    knownNoisestart2 = 1
    knownNoisestart3 = - 10 ** (- 12)
    knownNoisestart4 = 3 * 10 ** (- 10)
    knownNoisestart5 = -5 * 10 ** (- 7)


    # change of variables
    etastart, g2start = toEtaGamma(vol2start, a2start, deltabar)

    parastart = [etastart, g2start, knownNoisestart1, knownNoisestart2, knownNoisestart3, knownNoisestart4, knownNoisestart5]

    ## time set up
    # ts2 = np.zeros(ts.size)
    #tsdif = np.diff(ts)
    #ts2[0] = min(1 / ts[0], 20)
    #for i in range(ts2.size - 1):
    #    ts2[i + 1] = min(1 / tsdif[i], 20)
    ts2 = np.ones(ts.size)
    ts2[1:] =1/(1+ np.diff(ts))

    #timestart = time.time()

    param = minimize(lambda theta: MA1_triang_loglik(theta[0:2], np.diff(Y - (theta[2] + theta[3] * spread + theta[4] * volume + theta[5] * volLim + theta[6] * ts2) * info)),
                     parastart, method='Nelder-Mead',
                     options={'xtol': 1e-9, 'disp': False}).x

    #print("optim: "+totime(time.time()-timestart))
    #timestart = time.time()
    if returnVal:
        def l(theta): return MA1_triang_loglik(theta[0:2], np.diff(Y-(theta[2] + theta[3]*spread+ theta[4] * volume + theta[5] * volLim + theta[6] * ts2)*info))

        loglik = -l(param)
    #print(MA1_triang_loglik(param[0:2], np.diff(Y - (param[2] + param[3] * spread + param[4] * volume + param[5] * volLim + param[6] * ts2) * info)))
    # change back the variables
    etamle = param[0]
    g2mle = param[1]
    knownNoisemle1 = param[2]
    knownNoisemle2 = param[3]
    knownNoisemle3 = param[4]
    knownNoisemle4 = param[5]
    knownNoisemle5 = param[6]
    # etamle2 = param2[0]
    # g2mle2 = param2[1]
    volmle, a2mle = fromEtaGamma(etamle, g2mle, deltabar)
    IVmle = D * volmle

    #print("others: " + totime(time.time() - timestart))
    if returnVal:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, knownNoisemle3, knownNoisemle4, knownNoisemle5, etamle, g2mle,loglik]
    else:
        return [IVmle, a2mle, knownNoisemle1, knownNoisemle2, knownNoisemle3, knownNoisemle4, knownNoisemle5, etamle,
                g2mle]


#def QMLE_package(Y, D):
#    try:
#        model = ARMA(np.diff(Y), (0, 1))
#        etamle = model.fit(method='mle', trend='nc', disp=-1).params[0]
#        deltabar = float(D)/(Y.size-1)
#        g2mle = np.mean(np.diff(Y)**2)/(1+etamle**2)
#        Volmle = D*g2mle*(1+etamle)**2/deltabar
#        a2mle = -g2mle*etamle
#        return [Volmle, a2mle, etamle, g2mle]
#    except:
#        return [-1.05, -1.05, -1.05, -1.05]

###########################  TSRV related functions
def TSRV(Y, K, J=1):
    ## Compute TSRV estimator given tuning parameters K and J
    # =========================================================================
    # INPUT  - Y:      log(price) series, NOT raw prices
    #        - K & J:  tunning parameter
    #                  default take J = 1 (J<<K required)
    # OUTPUT - Volest: estimate for integrated volatility

    if K>J:

        n = Y.size
        nbarK = (n-K+1)/float(K)
        nbarJ = (n-J+1)/float(J)
        Volest = ARV(Y, K)-nbarK/nbarJ*ARV(Y, J)

        return Volest

    else:
        print('Error : J should be smaller than K')

def SW_TSRV(Y, K, J=1):
    ## Compute the sample-weighted TSRV estimator
    # =========================================================================
    # INPUT  - Y:      log(price) serise, NOT raw prices
    #        - K & J:  tunning parameter
    #                  default take J = 1 (J<<K required)
    # OUTPUT - Volest: estimate for integrated volatility

    Volest = ARV(Y, K)-1/float(K)*(ARV(Y[K:(-K)], J)+.5*ARV(Y[0:K], J)+.5*ARV(Y[(-K):-1], J))

    return Volest

###Kernel functions dictionnary for the Realized Kernels (RK)

fun_dict_RK = {"const1": [lambda x: 1, 1, 0, 0],
               "cubic": [lambda x: 1-3*x**2+2*x**3, 0.371, 1.2, 12.0],
               "th1": [lambda x: np.sin(np.pi/2*(1-x))**2, 0.375, 1.23, 12.1],
               "th2": [lambda x: np.sin(np.pi/2*(1-x)**2)**2, 0.219, 1.71, 41.7],
               "th16": [lambda x: np.sin(np.pi/2*(1-x)**16)**2, 0.032, 10.26, 14374.0]}


###Kernel functions dictionnary for the Pre-Averaging (PAE)
def triang_ker(x): return 2*(min(x, 1-x))


fun_dict_PA = {"triang_ker": (triang_ker, 1/3., 4, 151./5040, 1./6, 8./3)}


#################################### Pre-Averaged estimator (PAE) related functions

#####Basic functions  See (2.5) in Jacod and Mykland
def phi(k, f, known_fun=""):
    if known_fun == "" or k<100:
        return sum([f(float(i)/k)**2 for i in range(1, k+1)])
    else:
        return k*fun_dict_PA[known_fun][1]  # limit for large k


def phi_prime(k, f, known_fun=""):
    if known_fun == "" or k<100:
        return sum([(f(float(i)/k)-f(float(i-1)/k))**2 for i in range(1, k+1)])
    else:
        return 1./k*fun_dict_PA[known_fun][2]  # limit for large k

######Basic functions See (2.7) in Jacod and Mykland

def bar(dZ, i, k, g, g_vec=np.array([])):
    ##Compute the convolution Z-bar in (2.7) of a vector Z
    if g_vec.size == k-1 and k>1:  # allows faster computation
        return sum(g_vec*dZ[i:(i+k-1)])

    return sum(np.array([g(float(j)/k) for j in range(1, k)])*dZ[i:(i+k-1)])


def hat(dZ_square, i, k, g, dg_vec=np.array([])):
    ##Compute the convolution Z-hat in (2.7) of a vector Z
    if dg_vec.size == k and k>1:  # allows faster computation
        return sum(dg_vec*dZ_square[(i-1):(i+k-1)])

    return sum(np.array([(g(float(j)/k)-g(float(j-1)/k))**2 for j in range(1, k+1)])*dZ_square[(i-1):(i+k-1)])

##The pre-average estimator function for IV
def C_hat(Z, delta, k, g, T, known_fun=""):
    ##Compute the pre-average estimator for IV of parameters (delta,k,g) as defined in (2.8)
    dZ = np.diff(Z)
    dZ_square = dZ**2
    n = dZ.size
    g_vec = np.array([g(float(j)/k) for j in range(1, k)])
    dg_vec = np.array([(g(float(j)/k)-g(float(j-1)/k))**2 for j in range(1, k+1)])
    # import pdb;pdb.set_trace()
    return 1./(phi(k, g, known_fun)*(1-k*delta/T))*sum(
        [bar(dZ, i, k, g, g_vec)**2-.5*hat(dZ_square, i, k, g, dg_vec) for i in range(1,n-k+2)])

def V_hat(Z, delta, k, g, T, known_fun=""):

    dZ = np.diff(Z)
    dZ_square = dZ**2
    g_vec = np.array([g(float(j)/k) for j in range(1, k)])
    dg_vec = np.array([(g(float(j)/k)-g(float(j-1)/k))**2 for j in range(1, k+1)])

    phi_g = phi(k, g, known_fun)
    phi_prime_g = phi_prime(k, g, known_fun)
    C_phi_g = C_phi(k, g)
    C_phi_prime_g = C_phi_prime(k, g)
    C_phi_second_g = C_phi_second(k, g)

    coeff = 1./(phi_g**2*(1-k*delta/T))
    term1 = sum([4*C_phi_g/(3*phi_g**2)*bar(dZ, i, k, g, g_vec)**4 for i in range(int(T/delta)-k)])
    term2 = sum([4*(C_phi_prime_g/(phi_g*phi_prime_g)-C_phi_g/phi_g**2)*bar(dZ, i, k, g,
                                                                            g_vec)**2*hat(dZ_square,
                                                                                          i, k, g,
                                                                                          dg_vec) for
                 i in range(int(T/delta)-k)])
    term3 = sum([(C_phi_g/phi_g**2-2*C_phi_prime_g/(
        phi_g*phi_prime_g)+C_phi_second_g/phi_prime_g**2)*hat(dZ_square, i, k, g, dg_vec)**2 for i in
                 range(int(T/delta)-k)])

    return coeff*(term1+term2+term3)

##The PA estimator function for IQ
def Q_hat(Z, delta, k, g, T, known_fun=""):
    ##Compute the pre-average estimator for IQ of parameters (delta,k,g) as defined in (2.12)
    # in the form (term1+term2+term3)
    theta = k*np.sqrt(delta)

    dZ = np.diff(Z)
    dZ_square = dZ**2
    n = dZ.size
    # import pdb; pdb.set_trace()
    g_vec = np.array([g(float(j)/k) for j in range(1, k)])

    phi_int_g = phi_int(g, g, 0, "gg", known_fun)
    phi_int_gprime = phi_int(g, g, 0, "gprimegprime", known_fun)

    z_bar = np.array([bar(dZ, i, k, g, g_vec) for i in range(n-k+1)])

    term1 = 1/(3*theta**2*phi_int_g**2)*np.sum(z_bar**4)
    term2 = -delta*phi_int_gprime/(theta**4*phi_int_g**2)*np.sum(
        z_bar[:-k]**2*np.array([np.sum(dZ_square[(i+k):(i+2*k-1)]) for i in range(z_bar[:-k].size)]))
    term3 = delta*phi_int_gprime**2/(4*theta**4*phi_int_g**2)*np.sum(dZ_square[:-2]*dZ_square[2:])

    return term1+term2+term3


####Capital Phi functions
def C_phi(k, g):
    return sum([sum(
        [sum([g(float(j)/k)*g(float(j-i)/k)*g(float(l)/k)*g(float(l-i)/k) for l in range(k)]) for j in
         range(k)]) for i in range(k)])

def C_phi_prime(k, g):
    return sum([sum([sum([g(float(j)/k)*g(float(j-i)/k)*(g(float(l)/k)-g(float(l-1)/k))*(
        g(float(l-i)/k)-g(float(l-i-1)/k)) for l in range(k)]) for j in range(k)]) for i in range(k)])

def C_phi_second(k, g):
    return sum([sum([sum([(g(float(j)/k)-g(float(j-1)/k))*(g(float(j-i)/k)-g(float(j-i-1)/k))*(
        g(float(l)/k)-g(float(l-1)/k))*(g(float(l-i)/k)-g(float(l-i-1)/k)) for l in range(k)]) for
                     j
                     in range(k)]) for i in range(k)])

def phi_int(f, g, t, mode="", known_fun="", k=200):
    if mode == "gg" and t == 0:
        return fun_dict_PA[known_fun][1]

    if mode == "gprimegprime" and t == 0:
        return fun_dict_PA[known_fun][2]

    return sum([f(float(i)/k-t)*g(float(i)/k) for i in range(k)])

def C_phi2(f, g, mode="", known_fun="", k=200):
    if known_fun == "" or mode == "":
        return sum([phi_int(f, f, float(i/k))*phi_int(g, g, float(i/k)) for i in range(k)])
    else:
        if mode == "gg":
            return fun_dict_PA[known_fun][3]
        elif mode == "ggprime":
            return fun_dict_PA[known_fun][4]
        elif mode == "gprimegprime":
            return fun_dict_PA[known_fun][5]

def Psi(f, f_prime, known_fun="", k=200):
    phi_ffprime = C_phi2(f, f_prime, "ggprime", known_fun, k)
    phi_ff = C_phi2(f, f, "gg", known_fun, k)
    phi_fprimefprime = C_phi2(f_prime, f_prime, "gprimegprime", known_fun, k)

    return (np.sqrt(phi_ffprime**2+3*phi_ff*phi_fprimefprime)-phi_ffprime)/(3*phi_fprimefprime)

def alpha_avar(f, f_prime, known_fun="", k=200):
    phi_ffprime = C_phi2(f, f_prime, "ggprime", known_fun, k)
    phi_ff = C_phi2(f, f, "gg", known_fun, k)
    phi_fprimefprime = C_phi2(f_prime, f_prime, "gprimegprime", known_fun, k)
    phi_int_f = phi_int(f, f, 0, "gg", known_fun)

    num = 6*phi_ff*phi_fprimefprime+2*phi_ffprime*np.sqrt(
        phi_ffprime**2+3*phi_ff*phi_fprimefprime)-2*phi_ffprime**2
    den = 3**1.5*phi_int_f**2*np.sqrt(phi_fprimefprime)*np.sqrt(
        np.sqrt(phi_ffprime**2+3*phi_ff*phi_fprimefprime)-phi_ffprime)

    return num/den

# Step 1 of the preaveraging procedure
def pre_est(Z, delta, g, T, a=.5, w=5./6, known_fun=""):
    dZ = np.diff(Z)
    dZ_square = dZ**2
    l = min(T/delta**w, T/delta)
    k = int(max(min(np.sqrt(T/delta), (1-a)*l), 2.))

    l_prime = int(a*l)

    L = int(T/(l*delta))

    c = np.zeros(L)
    gamma = np.zeros(L)

    g_vec = np.array([g(float(j)/k) for j in range(1, k)])
    dg_vec = np.array([(g(float(j)/k)-g(float(j-1)/k))**2 for j in range(1, k+1)])

    for r in range(1, L):
        # import pdb; pdb.set_trace()
        c[r] = 1./(l_prime*delta*phi(k, g, known_fun))*sum(
            [bar(dZ, i, k, g, g_vec)**2-.5*hat(dZ_square, i, k, g, dg_vec) for i in
             range(int(r*l-l_prime-k), int(r*l-k))])

        gamma[r] = .5/l_prime*np.sum(dZ_square[int(r*l-l_prime):(r*l-1)])

    return [c, gamma]

def theta_hat(Z, delta, g, g_prime, T, a=.5, w=5./6, known_fun=""):
    c, gamma = pre_est(Z, delta, g, T, a, w, known_fun)

    L = c.size

    theta = np.zeros(L)

    for r in range(1, L):
        if c[r]>0:
            theta[r] = np.sqrt(gamma[r]/(Psi(g, g_prime, known_fun)*c[r]))
        else:
            theta[r] = np.sqrt(T)

    return theta


# step 2
def preaverage(Z, delta, g, g_prime, T, a=.5, w=5/6, known_fun="", mode="adaptive", theta=()):
    dZ = np.diff(Z)
    dZ_square = dZ**2
    n = dZ.size

    if mode == "adaptive":  # Computes the adaptive as in the paper from Jacod and Mykland
        theta = theta_hat(Z, delta, g, g_prime, T, a, w, known_fun)

    L = theta.size

    l = min(T/delta**w, T/delta)
    k = np.zeros(L)
    #k[0] = min(max(min(1./delta**.5, (1-a)*l), 2.), np.sqrt(T/delta))

    for r in range(1, L):
        k[r] = min(max(2, min(l/2., theta[r]/np.sqrt(delta))), 3*np.sqrt(T/delta))

    k[0] = np.mean(k[1:])

    A = np.ones(L)
    J = np.zeros(L+1)  # WARNING ! size is L+1

    J[0] = 1
    for j in range(1, L):
        J[j] = int(j*l)+1

    J[L] = n-int(k[L-1])+2
    A[L-1] = float((J[L]-J[L-1]+k[L-1]))/(J[L]-J[L-1])

    g_vec_dict = {}
    dg_vec_dict = {}
    for j in range(L):
        k_temp = int(k[j])
        g_vec_dict[j] = np.array([g(float(i)/k_temp) for i in range(1, k_temp)])
        dg_vec_dict[j] = np.array([(g(float(i)/k_temp)-g(float(i-1)/k_temp))**2 for i in range(1, k_temp+1)])

    return (sum([A[j]/phi(int(k[j]), g, known_fun)*sum(
        [bar(dZ, i, int(k[j]), g, g_vec_dict[j])**2-.5*hat(dZ_square, i, int(k[j]), g, dg_vec_dict[j]) for i in
         range(int(J[j]), int(J[j+1]))]) for j in range(L)]), k)


##################################        Realized Kernels (RK) related functions
def c_star(kernel_type, rho):
    k_00, k_11, k_22 = fun_dict_RK[kernel_type][1:4]
    d = k_00*k_22/k_11**2
    return np.sqrt(rho*k_11/k_00*(1+np.sqrt(1+3*d/rho**2)))


def gam(dX, h):  # Doesnt take into account the border effects
    # input : dX is the log return vector, that is dX[i] contains X[i+1]-X[i], that is dX = np.diff(X)
    # h : window size
    # output : gamma_h(X) as presented in BARNDORFF-NIELSEN and al
    h = abs(h)
    if h>0:
        return sum(dX[h:]*dX[0:-h])
    else:
        return sum(dX**2)


def gam_border(dX, h):  # Takes into account the border effects if possible

    n = dX.size

    if h>0:
        return sum(dX[h:(n-h)]*dX[:(n-2*h)])
    elif h<0:
        return sum(dX[-h:(n+h)]*dX[-2*h:])
    else:
        return sum(dX**2)


def K_rk(dX, kernel_type, H):  # Doesnt take into account the border effects

    k = fun_dict_RK[kernel_type][0]
    return sum([2*k((h-1)/H)*gam(dX, h) for h in np.arange(1, H+1)])+gam(dX, 0)


def K_rk_border(dX, kernel_type, H, start=0, end=-1):  # Takes into account the border effects if possible
    k = fun_dict_RK[kernel_type][0]
    return sum(
        [k((h-1)/H)*(gam_border(dX[(start-h):(end+h)], h)+gam_border(dX[(start-h):(end+h)], -h)) for h
         in np.arange(2, H+1)])+gam_border(dX[(start-1):(end+1)], 1)+gam_border(dX[(start-1):(end+1)],
                                                                                -1)+gam(dX[start:end], 0)


def realized_kernel(X, T, kernel_type, optional_args, mode="oracle"):  # Doesnt take into account the border effects
    dX = np.diff(X)

    if mode == "oracle":
        omega2, Q, rho = optional_args
        xi2 = omega2/np.sqrt(T*Q)
        c = c_star(kernel_type, rho)
        n_obs = dX.size
        H = c*np.sqrt(xi2)*np.sqrt(n_obs)
        return K_rk(dX, kernel_type, int(H))

    if mode == "manual":
        H = optional_args[0]
        return K_rk(dX, kernel_type, int(H))


def realized_kernel_border(X, T, kernel_type, optional_args, mode="oracle", start=0,
                           end=-1):  # Takes into account the border effects if possible
    dX = np.diff(X)

    if mode == "oracle":
        omega2, Q, rho = optional_args
        xi2 = omega2/np.sqrt(T*Q)
        c = c_star(kernel_type, rho)
        n_obs = dX[start:end].size
        H = c*np.sqrt(xi2)*np.sqrt(n_obs)
        # import pdb; pdb.set_trace()

    if mode == "manual":
        H = optional_args[0]

    if (start>=H) and ((-end-1)>=H):
        return K_rk_border(dX, kernel_type, int(H), start, end)
    else:
        return K_rk(dX[start:end], kernel_type, int(H))


####Dependent noise robust Realized Kernel estimator related functions

def K_rk_depnoise(dX, kernel_type, H, gamma):  # Doesnt take into account the border effects

    k_base = fun_dict_RK[kernel_type][0]
    threshold = H ** (-gamma)

    def k(x):
        if abs(x) <= threshold:
            return 1
        else:
            return k_base(abs(x) - threshold)

    return sum([2*k((h-1)/H)*gam(dX, h) for h in np.arange(1, H+1)])+gam(dX, 0)

def K_rk_border_depnoise(dX, kernel_type, H, gamma, start=0, end=-1):  # Takes into account the border effects if possible, robust to dependent noise
    k_base = fun_dict_RK[kernel_type][0]
    threshold = H**(-gamma)
    def k(x):
        if abs(x) <= threshold:
            return 1
        else:
            return k_base(abs(x) - threshold)

    return sum(
        [k((h-1)/H)*(gam_border(dX[(start-h):(end+h)], h)+gam_border(dX[(start-h):(end+h)], -h)) for h
         in np.arange(2, H+1)])+gam_border(dX[(start-1):(end+1)], 1)+gam_border(dX[(start-1):(end+1)],

                                                                                -1)+gam(dX[start:end], 0)
def realized_kernel_border_depnoise(X, T, kernel_type, optional_args, mode="manual", start=0,
                           end=-1):  # Takes into account the border effects if possible
    dX = np.diff(X)

    if mode == "oracle":
        omega2, Q, rho, gamma = optional_args
        xi2 = omega2/np.sqrt(T*Q)
        c = c_star(kernel_type, rho)
        n_obs = dX[start:end].size
        H = c*np.sqrt(xi2)*np.sqrt(n_obs)
        # import pdb; pdb.set_trace()

    if mode == "manual":
        H, gamma = optional_args[:2]

    if (start>=H) and ((-end-1)>=H):
        return K_rk_border_depnoise(dX, kernel_type, int(H),gamma, start, end)
    else:
        return K_rk_depnoise(dX[start:end], kernel_type, int(H),gamma)

#################################     Useful statistical  tools
def studentBias(values, target, avar, n, rate=0.25):
    if isinstance(n,np.ndarray):
        return np.mean((n**rate)*(values-target)/np.sqrt(avar))

    return (n**rate)*np.mean((values-target)/np.sqrt(avar))


def studentStd(values, target, avar, n, rate=0.25):
    if isinstance(n, np.ndarray):
        return np.std((n**rate)*(values-target)/np.sqrt(avar))
    return (n**rate)*np.std((values-target)/np.sqrt(avar))


def studentRMSE(values, target, avar, n, rate=0.25):
    if isinstance(n, np.ndarray):
        return np.sqrt(np.mean((n**(2*rate))*(values-target)**2/avar))
    return (n**rate)*np.sqrt(np.mean((values-target)**2/avar))


def studentQuantile(values, target, avar, n, q, rate=0.25):
    return norm.cdf(np.percentile((n**rate)*(values-target)/np.sqrt(avar), q, interpolation='lower'))


def empLoss(values, target, avar_theo, n, q, rate=0.25):  # Emp MSE over the theoretical asymptotic variance
    if isinstance(n, np.ndarray):
        return np.mean((n**(2*rate))*(values-target)**2/avar_theo)
    return (n**(2*rate))*np.mean((values-target)**2/avar_theo)


def rmse(values, target):
    """
    :type values: ndarray
    :type target: ndarray
    """
    return np.sqrt(np.mean((values-target)**2))


def theoLoss(avar, avar_best):

    """
    Computes the mean ratio of asymptotic variance over the best (optimal) AVAR
    :param avar: AVAR
    :param avar_best: Optimal AVAR
    :return:  mean of AVAR/Optimal AVAR
    """
    return np.mean(avar/avar_best)

####Various AVAR functions
def avar_qmle(IV,IQ,T):
    return 5*T*IQ/np.sqrt(IV)+3*(IV**1.5)

def avar_rk(IV,IQ,T, kernel = "th2"):

    k0,k1,k2 = fun_dict_RK[kernel][1:4]

    rho = IV/np.sqrt(T*IQ)
    c = c_star(kernel,rho)

    return 4*(T*IQ)**0.75*(c*k0+ 2*rho*k1/c+ k2/c**3)


###Yingying estimation functions
def knownNoise_Yingying_roll(Y, info, D):
    ## Compute the explicative part parameter of the QMLE related to the no-error model where the explicative part follows Roll
    ## The closed formula can be found in (10) in Yingying et al. (2016)
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - knownNoisereg: regression estimate for the known noise parameter

    num = np.sum(np.multiply(np.diff(info),np.diff(Y)))
    den = np.sum(np.diff(info)**2)
    knownNoisereg=num/den
    return knownNoisereg

def volatility_Yingying_roll(Y, info, D, knownNoiseest):
    ## Compute the volatility parameter of the QMLE related to the no-error model where the explicative part follows Roll
    ## The closed formula can be found in ERV in (13) in Yingying et al. (2016)
    ## This is equal to "realized volatility" on the estimated price
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    #          knownNoiseest:   known noise parameter estimator
    # OUTPUT - volatilityest: volatility estimator
    Yest = Y - knownNoiseest*info
    ret = np.diff(Yest)
    volatilityest=np.sum(ret**2)
    return volatilityest

def volatility_Yingying_spread(Y, info, spread, D, knownNoiseest):
    ## Compute the volatility parameter of the QMLE related to the no-error model where the explicative part follows signed spread model
    ## The closed formula can be found in ERV in (13) in Yingying et al. (2016)
    ## This is equal to "realized volatility" on the estimated price
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    #          knownNoiseest:   known noise parameter estimator
    # OUTPUT - volatilityest: volatility estimator
    Yest = Y - knownNoiseest*info*spread
    ret = np.diff(Yest)
    volatilityest=np.sum(ret**2)
    return volatilityest

def knownNoise_Yingying_spread(Y, info, spread, D):
    ## Compute the known noise estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - knownNoisereg: regression estimate for the known noise parameter (to change!!!!)

    num = np.sum(np.multiply(np.diff(info*spread),np.diff(Y)))
    den = np.sum(np.diff(info*spread)**2)
    knownNoisereg=num/den
    return knownNoisereg

def knownNoise_Yingying_spreadCanonical(Y, info, spread, D):
    ## Compute the known noise estimator when the microstructure follows Roll
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    # OUTPUT - knownNoisereg: regression estimate for the known noise parameter (to change!!!!)

    l = len(Y[0])
    num1 = np.sum(np.multiply(np.diff(info*spread),np.diff(Y)))
    num2 = np.sum(np.multiply(info[0][0:(l-1)] * spread[0][0:(l-1)], np.diff(Y)))
    den = np.sum(np.diff(info*spread)**2) + np.sum((info[0][0:(l-1)]*spread[0][0:(l-1)])**2)
    knownNoisereg1=num1/den
    knownNoisereg2 = num2 / den
    return [knownNoisereg1, knownNoisereg2]


def volatility_Yingying_Noise_roll(Y, info, D, knownNoiseest):
    ## Compute the closed formula of the volatility estimator (25) in Yingying et al. (2016)
    # =========================================================================
    # INPUT  - Y:      log(price) series
    #          D:      length of the sampling interval
    #          info:   trade indicator series
    #          knownNoiseest:   known noise parameter estimator
    # OUTPUT - volatilityest: volatility estimator
    Yest = Y - knownNoiseest*info
    ret = np.diff(Yest)
    volatilityest=np.sum(ret**2) + 2*np.sum(ret[0:(ret.size-1)]*ret[1:ret.size])
    return volatilityest

### general statistical function
def ACF(Y, L):
    ## Compute the ACF of Y up to the lag L
    acf = np.zeros(L)
    for k in range(L):
        S = Y.size - k - 1
        Yshrink = np.zeros(S)
        Ydec = np.zeros(S)
        for i in range(S):
            Ydec[i] = Y[i + k]
            Yshrink[i] = Y[i]
        acf[k] = pearsonr(Yshrink, Ydec)[0]
    return acf

### compute the estimation using Instrumental Variable as in Chaker
def estimationIV(Y, info, spread, lag):
    ## Compute the estimated known parameter using IV as in Chaker where lag can be bigger than 1
    num = 0
    den = 0
    S = Y.size - lag - 1
    Yret = np.diff(Y)
    knownNoiseRet = np.diff(info*spread)
    for i in range(S):
        num = num + knownNoiseRet[i]*Yret[i+lag]
        den = den + knownNoiseRet[i]*knownNoiseRet[i+lag]
    knownNoiseparamestim=num/den
    return knownNoiseparamestim

def timevaryingintensity(t,T):
### Compute the time-varying intensity at time t for horizon T which corresponds to model 3 in numerical simulations
### in Statistical inference for the doubly stochastic Self-Exciting Process
### corresponds to alpha^-1 in papers Efficient variance reduction and LOB
### in whole generality, we could add betas in arguments
    beta1 = -0.84
    beta2 = -0.26
    beta3 = -0.39
    intensity = np.exp(beta1) + ((np.exp(beta2) + np.exp(beta3)) ** 2)*((t/T - np.exp(beta2)/(np.exp(beta2) + np.exp(beta3))) ** 2)
    return intensity


### test on time
#start_time = time.time()
#
#for i in range(10000):
#    E = np.random.normal(0, 1, 1000)
#print("--- %s seconds ---" % (time.time() - start_time))

#start_time = time.time()
#for i in range(1000000):
#    E = np.random.normal(0, 1, 1)
#print("--- %s seconds ---" % (time.time() - start_time))

#start_time = time.time()
#E = np.random.normal(0, 1, 1000000)
#for i in range(1000000):
#    E1 = E[i]
#print("--- %s seconds ---" % (time.time() - start_time))

