import numpy as np
from pandas import DataFrame, Series
#from path import path


class Simulator(object):

    def __init__(self,nbsecond=1, M=500,d=1, dailySeconds = 23400, T=1.0, name = "MyAwesomeSimulator", regular_obs=1):
        # in whole generality we should provide in arguments the function to be used with non regular observations
        self.empty = True
        #self.default_path = path('/users/Simon/desktop/Local MLE vol code/results/')
        self.nbsecond=nbsecond
        self.M = M
        self.d=d
        self.dailySeconds=dailySeconds
        self.T = T
        self.delta = T/dailySeconds*nbsecond
        self.name = name
        self.regular_obs = regular_obs

        ##init to a "zero model"
        #self.simulGlobal(self.simulCV,[0],self.simulNoNoise,[],True)

    def simulObsTimes(self):
    # in whole generality we should provide in arguments the function to be used with non regular observations. For now we use a function from _aux.py
    # Goal: Simulate the observation times
    # Prerequisite: regular_obs=0
    # Output: write directly the observation times (and the number of observations) on the structure, and also return them
        if self.regular_obs == 1:
            print("You are trying to simulate observation times although initiated as regular")
        self.Nmax = 22000 # could be put as an argument in __init__
        self.obsTimes = np.zeros((M,self.Nmax+1))
        self.nbObs = np.zeros(M)
        exp = np.random.exponential(2*self.T/self.dailySeconds, (M,self.Nmax+1))
        for i in range(M):
            tcurrent = 0
            count = 0
            tcurrent = tcurrent + timevaryingintensity(tcurrent, self.T) * exp[i][count]
            while tcurrent < self.T:
                count = count+1
                self.obsTimes[i][count] = tcurrent
                tcurrent = tcurrent + timevaryingintensity(tcurrent, self.T) * exp[i][count]
            self.nbObs[i] = count+1
        return self.obsTimes

    def simulCV(self,priceArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        # *************************************************************************
        # Simulate the constant volatility model dX = sigma dW
        # *************************************************************************
        # INPUT:  1. sigma (first component of priceArgs)   - volatility
        #         2. nbsecond - # length of equi-distance time interval (second)
        #         3. M        - # of Monte Carlo Simulation
        #         4. d        - # of business day(s)
        #
        # *************************************************************************
        # OUTPUT: 1. (X,V)         - the process and the realized volatility^2
        if self.regular_obs==1:
            dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
            n=d*dnsimul #nb of obs of logreturns
            deltabar = T/dnsimul # 'dt' entre deux obs ("T time" )
            nsimul  = n + 1        # # of total simu
            sigma = priceArgs[0]
            W = np.sqrt(deltabar)*sigma*np.random.normal(0,1,(M,nsimul))
            return (np.cumsum(W, axis=1), sigma ** 2 * np.ones((M, nsimul)))
        if self.regular_obs==0:
            sigma = priceArgs[0]
            deltabarloc = np.zeros((M,self.Nmax+1))
            deltabarloc[0:M,0:self.Nmax] = np.sqrt(abs(np.diff(self.obsTimes)))
            W = deltabarloc * sigma * np.random.normal(0, 1, (M, self.Nmax+1))
            return np.cumsum(W, axis=1), sigma ** 2 * np.ones((M, self.Nmax+1))


    def simulDeterministicV(self, priceArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        # *************************************************************************
        # Simulate the deterministic volatility model dX = sigma(s) dW
        # *************************************************************************
        # INPUT:  1. sigma (first component of priceArgs)  - daily volatility, should be a function [0,1] -> positive real
        #         2. nbsecond - # length of equi-distance time interval (second)
        #         3. M        - # of Monte Carlo Simulation
        #         4. d        - # of business day(s)
        #
        # *************************************************************************
        # OUTPUT: 1. (X,V)         - the process and the realized volatility^2

        ####Clock
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        n=d*dnsimul #nb of obs of logreturns
        deltabar = T/dnsimul # 'dt' entre deux obs ("T time" )
        nsimul  = n + 1        # # of total simu


        sigma = priceArgs[0]
        global_sigma = np.array([sigma(float(i%dnsimul)/dnsimul) for i in range(nsimul)])
        #import pdb; pdb.set_trace()
        W = np.sqrt(deltabar)*global_sigma*np.random.normal(0,1,(M,nsimul))
        return (np.cumsum(W,axis =1), global_sigma**2*np.ones((M,nsimul)))

    def simulDeterJumpV(self, priceArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond, self.M, self.d, self.dailySeconds, self.T]
    # *************************************************************************
    # Simulate the deterministic volatility with jump in vol model dX = sigma(s) dW
    # *************************************************************************
    # INPUT:  1. sigma (first component of priceArgs)  - daily volatility, should be a function [0,1] -> positive real
    #         2. nbsecond - # length of equi-distance time interval (second)
    #         3. M        - # of Monte Carlo Simulation
    #         4. d        - # of business day(s)
    #         5. Poisson intensity
    #         6. jump size # proportion of sigma(t-), positive -> upside jump, negative, downside jump (sigma(t) = sigma(t-)(1+jumpSize) )
    # *************************************************************************
    # OUTPUT: 1. (X,V)         - the process and the realized volatility^2

        ####Clock
        dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
        n = d * dnsimul  # nb of obs of logreturns
        deltabar = T / dnsimul  # 'dt' entre deux obs ("T time" )
        nsimul = n + 1  # # of total simu

        sigma, intensity, jump_size_std = priceArgs

        global_sigma = np.array([sigma(float(i % dnsimul) / dnsimul) for i in range(nsimul)])

        mean_IV = np.sqrt(np.mean(global_sigma**2))
        jumpSize = jump_size_std*np.random.normal(0,1,(M,nsimul-1))*mean_IV

        dglobal_sigma = np.diff(global_sigma)

        J = np.random.binomial(1,intensity*deltabar,(M,nsimul-1))*jumpSize

        real_sigma = np.ones((M,nsimul))*global_sigma[0]
        real_sigma[:,1:] += np.cumsum(dglobal_sigma+J*global_sigma[:-1],axis=1)

        if real_sigma[real_sigma <0].size >0 :
            print("Warning, some volatility paths were negative due to jumps")

        # import pdb; pdb.set_trace()

        W = np.sqrt(deltabar) * real_sigma * np.random.normal(0, 1, (M, nsimul))
        return (np.cumsum(W, axis=1), real_sigma ** 2 * np.ones((M, nsimul)))

    def simulSVHeston(self, priceArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
    # *************************************************************************
    # Simulate the Heston + seasonal U-shape vol (see Andersen 2012) stochastic volatility model
    # *************************************************************************
    # INPUT:  1. nbsecond - # length of equi-distance time interval (second)
    #         2. a2true, mu, kappa, alpha, gamma, rho, lambd, A,B,C,a,b   -  parameters of the Heston
    #         3. M        - # of Monte Carlo Simulation
    #         4. d        - # of business day(s)
    #         5. mode     - List containing the indices of Y in outpout (subset of {1,2,3})
    #         6. getPaths - Boolean enabling to get the sample paths of the logprice and the volatility
    # *************************************************************************
    # OUTPUT: 1. (logPrice,volatility)         - the process and the realized volatility^2

    ## Set the parameter ******************************************************
    # SV - U Model as in Andersen and al 2012 model 2
    # dX        = mu dt+  sigma_u sigma_sv dW
    # dsigma_sv^2   = kappa(alpha - sigma_sv^2) dt + gamma sigma_sv dB + sigma_sv J dN
    # sigma_u(t) = C + Ae^-at + Be^-b(1-t)
    # corr(W,B) = rho
    # N         = Poisson(lambd)



    # - true parameters for the SV part of the volatility process ****************************


        #import pdb;pdb.set_trace()
        mu, kappa, alpha, gamma, rho, sigma_fun, jump_param, jump_mode = priceArgs
        X0 = np.log(100) # opening price

        ####Clock
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        n=d*dnsimul #nb of obs of logreturns
        deltabar = T/dnsimul # 'dt' entre deux obs ("T time" )
        nsimul=0
        if self.regular_obs == 1:
            nsimul  = n + 1        # # of total simu
        if self.regular_obs == 0:
            nsimul = self.Nmax + 1
            dnsimul= nsimul - 1

        logPrice = np.zeros((M,nsimul))
        volatility = np.zeros((M,nsimul))

        ## positive part of squareroot function
        def positive(x):
            if x >=0:
                return np.sqrt(x)
            else:
                return 0
        positive = np.vectorize(positive)

        deltasimul = np.zeros((M,nsimul-1))
        if self.regular_obs == 1:
            deltasimul += deltabar
        if self.regular_obs == 0:
            deltasimul += np.diff(self.obsTimes)
            for i in range(M):
                deltasimul[i][self.nbObs[i]]=0
        X    = np.zeros(nsimul)
        V_CIR = np.zeros(nsimul)
        V    = np.zeros(nsimul)


        Sigma_u = np.array([ sigma_fun(float(i)/dnsimul) for i in range(dnsimul+1)])

        ##Jumps
        J = np.zeros((M,nsimul))
        if jump_mode == "unif":
            idx_jump = np.random.uniform(jump_param[1],jump_param[2],M)*nsimul

            for iMC in range(M):
                J[iMC,int(idx_jump[iMC]):] = jump_param[0]

        elif jump_mode == "poisson": #DO NOT USE, NOT COMPATIBLE FOR NOW
            intensity = jump_param[0]
            nb_jumps = np.random.poisson(intensity,M)

            for iMC in range(M):
                idx_jump = np.array([int(np.random.uniform(0,1)*nsimul for k in range(nb_jumps[iMC]))])
                J[iMC,idx_jumps] = 1

            J = np.cumsum(J,axis=1)


        ## Simulation of the Process
        for iMC in range(M):
            if iMC%10 == 0 :
                print(iMC)

            V0 = np.random.gamma(2*kappa*alpha/gamma**2,gamma**2/(2*kappa),1) # sample the initial value from volatility process

            B  = np.random.normal(0,1,nsimul)
            W  = rho*B+positive(1-rho**2)*np.random.normal(0,1,nsimul)

            X *= 0
            V_CIR *=0
            V *= 0

            X[0] = X0
            V_CIR[0] = V0
            V[0] = V0*Sigma_u[0]**2

            #Euler scheme for the Heston
            for k in range(1,nsimul):
                X[k] = X[k-1] + mu*deltasimul[iMC][k-1] + positive(V[k-1]*deltasimul[iMC][k-1])*W[k]
                V_CIR[k] = V_CIR[k-1] + kappa*(alpha - V_CIR[k-1])*deltasimul[iMC][k-1]+ gamma*positive(V_CIR[k-1]*deltasimul[iMC][k-1])*B[k]
                V[k] = ((Sigma_u[k%dnsimul]- J[iMC,k]*Sigma_u[int(idx_jump[iMC])%dnsimul])**2)*V_CIR[k]#- J[iMC,k]*V[idx_jump[iMC]]

            logPrice[iMC,:] = X #Recopie
            volatility[iMC,:] = V

        outList = []

        outList.append(logPrice)
        outList.append(volatility)

        return outList

    def simulSV2F(self, priceArgs):
        ##Simulate the 2 arithmetic factor model as in Andersen et al. (2012)
        nbsecond, M, d, dailySeconds, T = [self.nbsecond, self.M, self.d, self.dailySeconds, self.T]

        mu, beta0,beta1,beta2,alpha1,alpha2,phi,rho1,rho2, sigma_fun, jump_param,jump_mode= priceArgs
        X0 = np.log(100)  # opening price

        ####Clock
        dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
        n = d * dnsimul  # nb of obs of logreturns
        deltabar = T / dnsimul  # 'dt' entre deux obs ("T time" )
        nsimul = n + 1  # # of total simu

        logPrice = np.zeros((M, nsimul))
        volatility = np.zeros((M, nsimul))

        ## positive part of squareroot function
        def positive(x):
            if x >= 0:
                return np.sqrt(x)
            else:
                return 0

        positive = np.vectorize(positive)

        deltasimul = np.zeros(n) + deltabar

        X = np.zeros(nsimul)
        V_SV1 = np.zeros(nsimul)
        V_SV2 = np.zeros(nsimul)
        V = np.zeros(nsimul)

        Sigma_u = np.array([sigma_fun(float(i) / dnsimul) for i in range(dnsimul + 1)])

        ##Jumps
        J = np.zeros((M, nsimul))
        if jump_mode == "unif":
            idx_jump = np.random.uniform(0, 1, M) * nsimul

            for iMC in range(M):
                J[iMC, int(idx_jump[iMC]):] = jump_param[0]

        elif jump_mode == "poisson":  # DO NOT USE, NOT COMPATIBLE FOR NOW
            intensity = jump_param[0]
            nb_jumps = np.random.poisson(intensity, M)

            for iMC in range(M):
                idx_jump = np.array([int(np.random.uniform(0, 1) * nsimul for k in range(nb_jumps[iMC]))])
                J[iMC, idx_jumps] = 1

            J = np.cumsum(J, axis=1)

        ## Simulation of the Process
        for iMC in range(M):
            # print(iMC)
            V01 = 1/np.sqrt(2*abs(alpha1))*np.random.normal(0, 1,
                                 1)  # sample the initial value from volatility process

            V02 = 0#1/np.sqrt(2*alpha2)*np.random.normal(0, 1,
                                 #1)  # sample the initial value from volatility process

            B = np.random.normal(0, 1, nsimul)
            W1 = rho1 * B + positive(1 - rho1 ** 2) * np.random.normal(0, 1, nsimul)
            W2 = rho2 * B + positive(1 - rho2 ** 2) * np.random.normal(0, 1, nsimul)

            X *= 0
            V_SV1 *= 0
            V_SV2 *= 0
            V *= 0

            X[0] = X0
            V_SV1[0] = V01
            V_SV2[0] = V02
            V[0] = np.exp(beta0+beta1*V_SV1[0] + beta2*V_SV2[0])* Sigma_u[0] ** 2

            # Euler scheme for the Heston
            for k in range(1, nsimul):
                X[k] = X[k - 1] + mu * deltasimul[k - 1] + positive(V[k - 1] * deltasimul[k - 1]) * B[k]
                V_SV1[k] = V_SV1[k - 1] + alpha1 *V_SV1[k - 1] * deltasimul[k - 1] +  positive(
                   deltasimul[k - 1]) * W1[k]
                V_SV2[k] = V_SV2[k - 1] + alpha2  * deltasimul[k - 1] +(1+phi *  V_SV2[k - 1])*positive(
                     deltasimul[k - 1]) * W2[k]
                V[k] = ((Sigma_u[k % dnsimul] - J[iMC, k] * Sigma_u[int(idx_jump[iMC]) % dnsimul]) ** 2) * np.exp(beta0+beta1*V_SV1[
                    k] + beta2*V_SV2[k]) # - J[iMC,k]*V[idx_jump[iMC]]

            logPrice[iMC, :] = X  # Recopie
            volatility[iMC, :] = V

        outList = []

        outList.append(logPrice)
        outList.append(volatility)

        return outList

    def simulPriceReplay(self,priceArgs): #priceArgs is useless here
        ##OUTPUT : Replay the scenario stored in self.X and self.V from a previous simulation
        return(self.X,self.V)

    def simulNoiseIID(self,noiseArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        nsimul = 0
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        n=d*dnsimul #nb of obs of logreturns
        nsimul  = n + 1        # # of total simu
        if self.regular_obs ==1:
            dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
            n = d * dnsimul  # nb of obs of logreturns
            nsimul = n + 1  # # of total simu
        if self.regular_obs == 0:
            nsimul = self.Nmax + 1
        a2true = noiseArgs[0]
        return np.sqrt(a2true)*np.random.normal(0,1,(M,nsimul))

    def simulNoiseIIDfromXi2(self,noiseArgs):

        ##WARNING : you should have already simulated the price process at this stage
        nbsecond, M, d, dailySeconds, T = [self.nbsecond, self.M, self.d, self.dailySeconds, self.T]
        ####Clock
        dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
        n = d * dnsimul  # nb of obs of logreturns
        nsimul = n + 1  # # of total simu

        xi2,start,end,T_obs = noiseArgs

        a2true = xi2/np.mean(1/np.sqrt(T_obs*(self.Q[:,end]-self.Q[:,start])))
        self.noiseArgs = [a2true]
        return np.sqrt(a2true) * np.random.normal(0, 1, (M, nsimul))

    def simulNoNoise(self,noiseArgs):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        #priceArgs is useless here
        ##OUTPUT : returns a null array
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        nsimul = d*dnsimul +1
        return np.zeros((M,nsimul))

    def simulNoiseReplay(self,noiseArgs):#noiceArgs is useless here
        ##OUTPUT : Replay the scenario stored in self.U from a previous simulation
        return self.U

    def explicativePartRollIID(self,knownNoiseArgs):
        # knownNoiseArgs corresponds to the parameter
        # of the trade type
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        ####Clock
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        n=d*dnsimul #nb of obs of logreturns
        nsimul  = n + 1        # # of total simu
        info = 2*np.random.binomial(n=1, p=.5, size=(M, nsimul))-1
        KU = knownNoiseArgs*info
        ## test with spread
        #for i in range(math.floor((n-10)/2)) :
        #    KU[(2*i)]=10*info[(2*i)]
        outList = []
        outList.append(KU)
        outList.append(info)
        return outList

    def explicativePartRollAuto(self, knownNoiseArgs, auto):
        # knownNoiseArgs corresponds to the parameter, 0 < auto < 1 to the coefficient of 1-lag autocorrelation
        # of the trade type
        nsimul=0
        nbsecond, M, d, dailySeconds, T = [self.nbsecond, self.M, self.d, self.dailySeconds, self.T]
        if self.regular_obs ==1:
            dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
            n = d * dnsimul  # nb of obs of logreturns
            nsimul = n + 1  # # of total simu
        if self.regular_obs == 0:
            nsimul = self.Nmax + 1
        info = np.zeros((M,nsimul))
        for i in range(M):
            info[i][0] = 2*np.random.binomial(n=1, p=.5, size=1)-1
            for j in range(nsimul-1):
                test = np.random.binomial(n=1, p=auto, size=1)
                if test==1:
                    info[i][j+1]=info[i][j]
                else:
                    info[i][j+1] = 2 * np.random.binomial(n=1, p=.5, size=1) - 1
        KU = knownNoiseArgs * info
        outList = []
        outList.append(KU)
        outList.append(info)
        return outList

    def explicativePartSpread(self, knownNoiseArgs, auto, meanHalfSpread, varHalfSpread, ARparam):
        # knownNoiseArgs corresponds to the parameter, 0 < auto < 1 to the coefficient of 1-lag autocorrelation
        # of the trade type, meanHalfSpread, varHalfSpread, ARparam to the coefficients to be used
        # to simulate the AR(1) model that the spread follows
        nbsecond, M, d, dailySeconds, T = [self.nbsecond, self.M, self.d, self.dailySeconds, self.T]
        dnsimul = int(dailySeconds / nbsecond)  # daily number of obs of logreturns
        n = d * dnsimul  # nb of obs of logreturns
        nsimul = n + 1  # # of total simu
        info = np.zeros((M,nsimul))
        for i in range(M):
            info[i][0] = 2*np.random.binomial(n=1, p=.5, size=1)-1
            for j in range(nsimul-1):
                test = np.random.binomial(n=1, p=auto, size=1)
                if test==1:
                    info[i][j+1]=info[i][j]
                else:
                    info[i][j+1] = 2 * np.random.binomial(n=1, p=.5, size=1) - 1
        # simulate the AR(1) for the (half-) spread
        Hspread = np.zeros((M, nsimul))
        for i in range(M):
            Hspread[i][0] = meanHalfSpread
            for j in range(nsimul-1):
                Hspread[i][j+1] = meanHalfSpread + ARparam*(Hspread[i][j] - meanHalfSpread) + np.random.normal(0, np.sqrt(varHalfSpread), 1)
        KU = knownNoiseArgs * Hspread * info
        outList = []
        outList.append(KU)
        outList.append(info)
        outList.append(Hspread)
        return outList

    def simulGlobal(self, priceFun, priceArgs, noiseFun, noiseArgs, knownNoiseArgs, infotype = "none", recordMode = False):
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        ####Clock
        dnsimul = int(dailySeconds/nbsecond) #daily number of obs of logreturns
        n=d*dnsimul #nb of obs of logreturns
        deltabar = T/dnsimul # 'dt' entre deux obs ("T time" )
        nsimul  = n + 1        # # of total simu

        X,V = priceFun(priceArgs)

        ## Integrated quantities by Riemann sums *******************************
        IV = np.zeros((M, nsimul))
        R = np.zeros((M, nsimul))
        Quarticity = np.zeros((M, nsimul))
        if self.regular_obs == 1: #haven't coded it yet for the non regular case
            IV[:, 1:] = np.cumsum(V * deltabar, axis=1)[:, :-1]  # integrated volatility
            R[:, 1:] = np.cumsum((V ** (1.5)) * deltabar, axis=1)[:, :-1]  # tricity
            Quarticity[:, 1:] = np.cumsum((V ** 2) * deltabar, axis=1)[:, :-1]  # quarticity

        ####Record
        if recordMode:
            self.nbsecond = nbsecond
            self.M = M
            self.d = d
            self.dailySeconds = dailySeconds
            self.T = T

            #self.X = X
            self.V = V
            #self.U = U

            self.IV = IV
            self.R = R
            self.Q = Quarticity
            self.empty = False

            self.noiseArgs = noiseArgs
            self.priceArgs = priceArgs

        U = noiseFun(noiseArgs)
        Y = X+U
        if infotype == "roll":
            KU,info = self.explicativePartRollAuto(knownNoiseArgs=knownNoiseArgs, auto= 0.3)# KU known part of the noise, Info information, auto could be put in argument of the funciton if needed
            Y = Y + KU
            if recordMode:
                self.KU = KU
                self.info = info
                self.knownNoiseArgs = knownNoiseArgs
        if infotype == "spread":
            KU,info,spread = self.explicativePartSpread(knownNoiseArgs=knownNoiseArgs, auto= 0.3, meanHalfSpread= 0.000125, varHalfSpread = 10 ** (-10), ARparam = 0.6)# KU known part of the noise, Info information, auto, meanHalfSpread, varHalfSpread, ARparam could be put in argument of the funciton if needed
            Y = Y + KU
            if recordMode:
                self.KU = KU
                self.info = info
                self.knownNoiseArgs = knownNoiseArgs
                self.spread=spread
        if recordMode :
            self.Y = Y


        outList = []

        outList.append(Y)
        outList.append(IV)
        outList.append(R)
        outList.append(Quarticity)

        return outList

    def rhoGlobal(self, start = 0, end = -1, recordMode=False):
        T_obs = self.d*self.T*(1-(start-end-1)/(self.IV[0,:].size))
        rhoGlo = (self.IV[:,end]-self.IV[:,start])/np.sqrt(T_obs*(self.Q[:,end]-self.Q[:,start]))

        if recordMode :
            self.rho_global = rhoGlo

        return rhoGlo

    def rhoLocal(self, nb_blocks): #careful

        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        T_local = T*d/nb_blocks
        ndaily = int(dailySeconds/nbsecond)

        rho_loc = np.zeros((M,nb_blocks))
        for iMC in range(M):
            rho_loc[iMC,:] = np.sqrt(1.0/T_local)*np.array([(self.IV[iMC,d*ndaily/nb_blocks*(i+1)]-self.IV[iMC,d*ndaily/nb_blocks*i])/np.sqrt(self.Q[iMC,d*ndaily/nb_blocks*(i+1)]-self.Q[iMC,d*ndaily/nb_blocks*i]) for i in range(nb_blocks)])

        return rho_loc

    def kappaGlobal(self,start = 0, end = -1, recordMode=False):
        T_obs = self.d * self.T * (1 - (start - end - 1) / (self.IV[0, :].size))
        kappaGlo = (self.R[:,end]-self.R[:,start])/(T_obs**0.25*((self.Q[:,end]-self.Q[:,start])**0.75))

        if recordMode :
            self.kappa_global = kappaGlo

        return kappaGlo

    def kappaLocal(self, nb_blocks):

        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        T_local = T*d/nb_blocks
        ndaily = int(dailySeconds/nbsecond)

        kappa_loc = np.zeros((M,nb_blocks))
        for iMC in range(M):
            kappa_loc[iMC,:] = 1.0/(T_local)**0.25*np.array([(self.R[iMC,d*ndaily/nb_blocks*(i+1)]-self.R[iMC,d*ndaily/nb_blocks*i])/(self.Q[iMC,d*ndaily/nb_blocks*(i+1)]-self.Q[iMC,d*ndaily/nb_blocks*i])**0.75 for i in range(nb_blocks)])

        return kappa_loc

    def avarQMLE(self,start = 0, end = -1):
        #Output : The asymptotic theoretic avar of the QMLE as in D.Xiu (2010)
        T_obs = self.d * self.T * (1 - (start - end - 1) / (self.IV[0, :].size))
        a2true = self.noiseArgs[0]
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]

        return 5*np.sqrt(a2true)*(self.Q[:,end]-self.Q[:,start])*T_obs/np.sqrt(self.IV[:,end]-self.IV[:,start]) + 3*np.sqrt(a2true)*(self.IV[:,end]-self.IV[:,start])**1.5 #Oracle sample path variance of the MN convergence of the QMLE

    def avarLQMLE(self,nb_blocks,start = 0, end = -1): #careful
        #Output : The asymptotic theoretic avar of the local QMLE with nb_blocks blocks
        a2true = self.noiseArgs[0]
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]
        T_local = T*d/nb_blocks
        ndaily = int(dailySeconds/nbsecond)

        avar_loc = np.zeros((M,nb_blocks))
        for iMC in range(M):
            avar_loc[iMC,:] = np.array([5*np.sqrt(a2true)*(self.Q[iMC,d*ndaily/nb_blocks*(i+1)]-self.Q[iMC,d*ndaily/nb_blocks*i])/((d*T_local)*np.sqrt(self.IV[iMC,d*ndaily/nb_blocks*(i+1)]-self.IV[iMC,d*ndaily/nb_blocks*i])) + 3*np.sqrt(a2true)*(self.IV[iMC,d*ndaily/nb_blocks*(i+1)]-self.IV[iMC,d*ndaily/nb_blocks*i])**1.5/((d*T_local)**2) for i in range(nb_blocks)])#Oracle sample path variance of the MN convergence of the LQMLE

        return avar_loc

    def avarRK(self,d_rk,k0,k1,start = 0, end = -1):
        T_obs = self.d * self.T * (1 - (start - end - 1) / (self.IV[0, :].size))
        a2true = self.noiseArgs[0]
        rho_global = self.rhoGlobal(start,end)
        nbsecond, M, d, dailySeconds, T = [self.nbsecond,self.M,self.d,self.dailySeconds,self.T]

        return np.sqrt(a2true)*(T_obs)**0.75*(self.Q[:,end]-self.Q[:,start])**0.75*(16/3)*np.sqrt(rho_global*k0*k1)*((1 / np.sqrt(1 + np.sqrt(1 + 3 * d_rk / rho_global))) + np.sqrt(1 + np.sqrt(1 + 3 * d_rk / rho_global)))

    def toCSV(self,path):

        print("Saving returns...")
        dfY = DataFrame(self.Y)
        dfY.to_csv(path+"Y.csv")

        print("saving IV...")
        dfIV = DataFrame(self.IV)
        dfIV.to_csv(path + "IV.csv")

        print("saving R...")
        dfR = DataFrame(self.R)
        dfR.to_csv(path + "R.csv")

        print("saving IQ...")
        dfQ = DataFrame(self.Q)
        dfQ.to_csv(path + "Q.csv")

        various_dict = { "empty": self.empty,
                         "default_path" : self.default_path,
                         "nbsecond" : self.nbsecond,
                         "M" : self.M,
                         "d" : self.d,
                         "dailySeconds" : self.dailySeconds,
                         "T": self.T,
                         "delta" : self.delta,
                         "NoiseArgs" : self.noiseArgs[0],
                         "Name" : self.name}

        print("saving Various parameters...")
        dfVarious = Series(various_dict)
        dfVarious.to_csv(path +"Various.csv" )

        print("Done !")

    def fromCSV(self,path):

        print("Loading returns...")
        self.Y = np.array(DataFrame.from_csv(path+"Y.csv"))
        print("Loading IV...")
        self.IV = np.array(DataFrame.from_csv(path + "IV.csv"))
        print("Loading R...")
        self.R = np.array(DataFrame.from_csv(path + "R.csv"))
        print("Loading IQ...")
        self.Q = np.array(DataFrame.from_csv(path + "Q.csv"))

        print("Loading various parameters...")
        various = Series.from_csv(path + "Various.csv")
        self.empty = various["empty"]
        self.default_path = various["default_path"]
        self.nbsecond = float(various["nbsecond"])
        self.M = float(various["M"])
        self.d = float(various["d"])
        self.dailySeconds = float(various["dailySeconds"])
        self.T = float(various["T"])
        self.delta = float(various["delta"])
        self.noiseArgs = [float(various["NoiseArgs"])]
        self.name = various["Name"]

        print("Done !...")

    def info(self):

        print("Name : {0}".format(self.name))
        print("Noise :{0}".format(self.noiseArgs[0]))
        print("M : {0}".format(self.M))
        print("no. obs. {0}".format(self.Y[0,:].size))
