#!/usr/bin/env python
# -*- coding: utf-8 -*-

import numpy as np
from pandas import DataFrame
#from ivol._aux import QMLE_package, realized_kernel_border, realized_kernel_border_depnoise, c_star, preaverage, Psi, TSRV, Q_hat, C_hat, alpha_avar, \
#    rmse, studentRMSE, studentBias, studentQuantile, studentStd, empLoss, theoLoss, fun_dict_PA, fun_dict_RK, avar_qmle, avar_rk

#Some references
_ref_qmle = "[Xiu, 2010] faculty.chicagobooth.edu/dacheng.xiu/research/QMLE1D.pdf"
_ref_rk = "[Barndorff-Nielsen et al., 2008] http://onlinelibrary.wiley.com/doi/10.3982/ECTA6495/abstract"
_ref_pae = "[Jacod et al., 2009] http://www.sciencedirect.com/science/article/pii/S0304414908001701"
_ref_tsrv = "[Zhang et al, 2005] https://www.jstor.org/stable/27590680 "
_ref_efficient_asymptotic = "[Clinet,Potiron 2017] https://arxiv.org/abs/1701.01185"


_ref_rho = _ref_efficient_asymptotic+", p.3 "

_ref_kappa = _ref_efficient_asymptotic+", p.5 "

_whatisdone = """
So far, the package enables to perform non-parametric estimation of :
- IV (Integrated volatility) :
    - QMLE (quasi-maximum likelihood estimator as in %(qmle)s), with the class Vol_QMLE
    - RK (realized kernels as in %(rk)s), with the class Vol_RK
    - PAE (pre-averaged estimator as in %(pae)s), with the class Vol_PA
    - TSRV (two-scale realized volatility as in %(tsrv)s), with the class Vol_TSRV
-IQ (Integrated quarticity) :
    - PAE (pre-averaged estimator as in %(pae)s), with the class Quart_PA

-Heteroskedastic measure Rho, as defined in %(rho)s
    - by pre-estimating the IV and the IQ with the class Rho_Direct

Use help() on one of the above classes to get more specific information.

The framework is the classical one of a noisy Ito process. The noise is assumed to be IID of
constant variance. The presence of jumps in the price process will imply that the estimators of IV
will estimate IV + sum(square jumps) instead. In other words, the present estimators are NOT robust
to jumps.

The package also offers simple estimators of the noise variance :

    - QMLE (quasi-maximum likelihood estimator as in %(qmle)s), with the class Noise_QMLE, or directly via Vol_QMLE
    - RV (realized variance), with the class Noise_realized

Finally, and maybe most important, the present package offers local methods in order to improve
the above mentioned estimators as in the methodology of %(clinetPotiron)s. you can use local estimations by creating a local class version
 using the function localize() as follows :

 >>>Local_Class = localize(One_Global_Class)

 and then define instances of this new class to perform local estimations.
 Use help() on the function localize for more info.

Future improvements  :
-MSRV estimator.
-Robustness to jumps.
-More general noise process.
-a consistent Tricity estimator ? and thus a kappa estimator.
-... (suggestions, comments or help are welcome !)


"""%{"rho": _ref_rho, "qmle": _ref_qmle, "rk":_ref_rk, "pae":_ref_pae, "tsrv":_ref_tsrv, "clinetPotiron" : _ref_efficient_asymptotic}



__doc__="""
    ivol is a package gathering various estimators
    for integrated volatility for high-frequency data.

    Original author : Simon Clinet.   simon (at) ms.u-tokyo.ac.jp, http://www.ms.u-tokyo.ac.jp/~simon/

    Current maintainer : Simon Clinet

    Example :

    from ivol.estim import Vol_QMLE
    import numpy as np

    sigma = np.sqrt(0.1) # annual volatility
    a2 = 0.0002**2 # noise variance
    T = 1/252 # One working day in annual time
    d=1 # One day
    daily_obs = 23400 # One observation evey second

    X = sigma*np.sqrt(T/daily_obs)*np.random.normal(0,1,daily_obs) #efficient log-returns
    U = np.sqrt(a2)*np.random.normal(0,1,daily_obs) #noise
    Y = X+U

    input_data = {"Y": Y, "T":T, "d":d}

    start, end = 0,-1 #start and end indices for the estimation procedure
    est = Vol_QMLE(start,end,input_data) #The QMLE estimator
    est.estim()
    est.info()

    ###A local QMLE on 8 blocks
    from ivol.estim import Vol_estimator, localize

    input_data_local = {"Y": Y, "T":T, "d":d, "b":8, "estClass": Vol_QMLE}

    Vol_local = localize(Vol_estimator)
    est_local = Vol_local(start,end,input_data_local)

    est_local.estim()
    est_local.info()

    ##################

    estim is a module that contains the class Estimator and various daughter
    classes containing useful estimation/information methods for Integrated volatility,
    integrated quarticity and so forth. aux contains auxiliary functions for the estimations (use/modify it at your own risk).

    The class architecture is as follows :

    One generic and abstract class Estimator.

    Various generic and abstract daughter methods inheriting from Estimator :

    - Vol_estimator, for integrated volatility.
    - Quart_estimator, for integrated quarticity.
    - Tric_estimator, for integrated tricity.
    - Rho_estimator, for the heteroskedastic measure rho.
    - Kappa_estimator, for the heteroskedastic measure kappa.
    - Noise_estimator, for the noise variance.

    Real estimation classes,inheriting from one of those abstract classes. they are the classes
    one should use in practice.

    %(sofar)s

"""%{'sofar':_whatisdone}

_class_init_mode_doc = """

Modes :
--------
Two modes are available for the creation of an estimator.

"fromDict" : default case. In that case the user should input an appropriate dictionnary
 containing at least the sample data Y, the horizon time T, and the number of days d as specified
 below.
 "fromValues": If you want directly to create an estimator with pre-computed results values. This
 is useful if one wants to beneficiate from the implemented methods to analyze the values. in
 that case  the user should provide an appropriate dictionary
 containing at least the values array, the horizon time T, and the number of days d as specified
 below.

"""
_class_param_doc = """

Parameters :
------------
to create an instance, one should provide the following parameters :
start : index to be used in the sample data representing the start point for the estimation
end : index to be used in the sample data representing the end point for the estimation. end
can be written as a positive index (and then end should be larger than start) or as a negative index
following  smart indexing rules of Python.
modeArg : should contain the input dictionary containing the relevant data.
    if mode is "fromDict", modeArg should contain the fields :
    -"Y" : the logreturns as a (M x n) array. M is the number of trajectories. n is the number of obs for each trajectory
    -"T" : horizon time of one day, or one trajectory if d=1.
    -"d" : number of days per trajectory.
    -"IV" : (optional) the integrated volatility paths.
    -"Q" : (optional) the integrated quarticity paths.
    -"R" : (optional) the integrated tricity paths (the integrated third power of volatility).
    -"noiseVar" : (optional) the noise variance, a float positive number.

    if mode is "fromValues", modeArg should contain the fields :
    -"values": an one-dimensional float array containing the results of past estimations
    -"T" : horizon time of one day, or one trajectory if d=1.
    -"d" : number of day.

name : a str representing the name of your estimator.
mode : "fromDict" or "fromValues". see above.
"""

class Estimator(object):

    """
    Estimator is the most general structure to represent an estimating procedure and its results.
    The class Estimator itself is mostly virtual and only sets the formalism that should be used for more concrete
    estimator classes such as Vol_estimator, Quart_estimator, etc...
    It is not intended to be used to create instances.

    In particular, an Estimator object should always feature :
    - A 'values' field, containing the results of the estimation procedure.
    - 'start' and 'end' fields, which are selectors (int) to specify the start and end
      indices in the sample data that the estimator should use in estim()
    - An estim() method, that fills the values field.

    - A 'rate' field, representing the rate of convergence in the related CLT.

    The object can also contain oracle quantities in order to perform efficiency analysis of the
    estimator through simulations and so forth. In that case, additional fields and methods are

    - A 'target' field, containing the target values to be estimated
    - As much as possible, a valid compute_avar() method, filling the field 'avar' containing
      the asymptotic variance of the associated CLT.

    Use help on daughter classes to get more specific information.

    """

    def __init__(self, start, end, modearg, name="myEstimator", mode="fromDict"):
        self.start = start
        self.end = end
        self.mode = mode
        self.name = name
        self.estimMode = "default"
        self.sub_estimator = None
        self.avar = None
        self.best_avar = None
        self.target = None
        self.values = None
        self.rate = 0
        self.n_obs = 0
        self.n = 0

        if mode == "fromValues":
            if "values" in modearg:
                self.values = modearg["values"]
                self.M = self.values.size
                self.T = modearg["T"]
                self.d = modearg["d"]
            else:
                raise ValueError("no values field in modearg")

    def setParam(self, *param):
        """
        method for setting parameters that are specific to the estimator
        :param param: parameters of the esimator
        :type param : tuple
        :return: none
        """
        if param != ():
            self.estimMode = param[0]

    def estim(self, optionalArgs=()):
        """
        Estimation procedure. Fills the 'values' field.
        :param optionalArgs: optional arguments for the estimation
        :type optionalArgs: tuple
        :return: none
        """
        pass

    def compute_avar(self):
        """
        fills the 'avar' field, if the necessary oracle quantities are available.
        :return: none
        """
        pass

    def compute_best_avar(self):
        """
        fills the 'best_avar' field. avar_best represents the bound of efficiency in terms
        of asymptotic variance in the CLT.
        :return: none
        """
        pass

    def info(self):
        """
        A simple info printer function.
        :return: none
        """
        print("Estimator : {0}".format(self.name))
        print("Bias :{0}".format(self.bias()))
        print("Std:{0}".format(self.std()))
        print("RMSE:{0}".format(self.rmse()))
        print("--------------------------")

    def __getitem__(self, item):
        return self.values.__getitem__(item)

    def __repr__(self):
        answ = "Estimator :{0}\n".format(self.name)
        answ+= "Values :\n"
        answ+=self.values.__repr__()
        return answ

    def __len__(self):
        return len(self.values)

    def __add__(self, other):
        return self.values + other

    def __radd__(self, other):
        return self.values+other

    def __mul__(self, other):
        return self.values * other

    def __rmul__(self, other):
        return self.values*other

    def __sub__(self, other):
        return self.values-other

    def __rsub__(self, other):
        return other-self.values

    def __truediv__(self, other):
        return self.values/other

    def __rtruediv__(self, other):
        return other/self.values

    def __pow__(self, power, modulo=None):
        return self.values**power

    def std(self):
        return np.std(self.values)

    def mean(self):
        return np.mean(self.values)

    def var(self):
        return np.var(self.values)

    def bias(self):
        return np.mean(self.values-self.target)

    def rmse(self):
        return rmse(self.values, self.target)

    def stdr(self):
        return np.std(self.values/self.target)

    def meanr(self):
        return np.mean(self.values/self.target)

    def varr(self):
        return np.var(self.values/self.target)

    def biasr(self):
        return np.mean(self.values/self.target)-1

    def rmser(self):
        return rmse(self.values/self.target, 1)

    def studentBias(self):
        return studentBias(self.values, self.target, self.avar, self.n_obs, self.rate)

    def studentStd(self):
        return studentStd(self.values, self.target, self.avar, self.n_obs, self.rate)

    def studentRMSE(self):
        return studentRMSE(self.values, self.target, self.avar, self.n_obs, self.rate)

    def studentQuantile(self, q):
        return studentQuantile(self.values, self.target, self.avar, self.n_obs, q, self.rate)

    def empLoss(self):
        return empLoss(self.values, self.target, self.best_avar, self.n_obs, self.rate)

    def theoLoss(self):
        return theoLoss(self.avar, self.best_avar)

    def corr(self,y):
        if isinstance(y,Estimator):
             return self.corr(y.values)
        elif isinstance(y,np.ndarray):
             if y.size == self.values.size:
                 return np.corrcoef(np.array([self.values,y]))[0,1]
             else :
                 raise ValueError("the length of the argument and the length of self.values differ")
        else :
             raise ValueError("the argument should be a one dimensional numpy array")


    def toCsv(self, path):
        outputData = DataFrame()
        outputData["Target"] = self.target
        outputData["Values"] = self.values
        outputData["Avar"] = self.avar
        outputData["Best Avar"] = self.best_avar

        if path[-4:] == '.csv':
            outputData.to_csv(path[:-4]+self.name+"outputData.csv")
        else:
            outputData.to_csv(path+self.name+"outputData.csv")


class Vol_estimator(Estimator):

    __doc__="""
    A Vol_estimator object is designed to perform efficient estimations of Integrated Volatility. It inherits the
    basic structure of a general Estimator object. The class Vol_Estimator itself is abstract and is not intended to be used.

    %(mode)s

    %(param)s

    Use also : help() on Estimator class for more generic info
    """%{"mode":_class_init_mode_doc, "param": _class_param_doc}

    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)


        if mode == "fromValues":
            self.IV = np.zeros((self.M,2))
            self.Q = np.zeros((self.M,2))
            self.R = np.zeros((self.M,2))
            self.n = 2
            self.start = 0 #To avoid conflict
            self.end = -1
            self.T_obs = self.T

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape


            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            if "info" in modearg:
                if len(modearg["info"].shape) == 1:
                    self.info = modearg["info"].reshape((1, modearg["info"].shape[0]))
                else:
                    self.info = modearg["info"]

            if "volume" in modearg:
                if len(modearg["volume"].shape) == 1:
                    self.volume = modearg["volume"].reshape((1, modearg["volume"].shape[0]))
                else:
                    self.volume = modearg["volume"]

            if "ts" in modearg:
                if len(modearg["ts"].shape) == 1:
                    self.ts = modearg["ts"].reshape((1, modearg["ts"].shape[0]))
                else:
                    self.ts = modearg["ts"]

            if "spread" in modearg:
                if len(modearg["spread"].shape) == 1:
                    self.spread = modearg["spread"].reshape((1, modearg["spread"].shape[0]))
                else:
                    self.spread = modearg["spread"]

            if "knownNoiseArgs" in modearg:
                self.knownNoiseArgs = modearg["knownNoiseArgs"]
            else:
                self.knownNoiseArgs = 0

            if "volLim" in modearg:
                if len(modearg["volLim"].shape) == 1:
                    self.volLim = modearg["volLim"].reshape((1, modearg["volLim"].shape[0]))
                else:
                    self.volLim = modearg["volLim"]

            if "OPI" in modearg:
                if len(modearg["OPI"].shape) == 1:
                    self.OPI = modearg["OPI"].reshape((1, modearg["OPI"].shape[0]))
                else:
                    self.OPI = modearg["OPI"]

            if "nbObs" in modearg:
                self.nbObs = modearg["nbObs"]
            else:
                self.nbObs = 0

            if "obsTimes" in modearg:
                if len(modearg["obsTimes"].shape) == 1:
                    self.obsTimes = modearg["obsTimes"].reshape((1, modearg["obsTimes"].shape[0]))
                else:
                    self.obsTimes = modearg["obsTimes"]

            if "Nmax" in modearg:
                self.Nmax = modearg["Nmax"]
            else:
                self.Nmax = 0

            if "regular_obs" in modearg:
                self.regular_obs = modearg["regular_obs"]
            else:
                self.regular_obs = 1

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            if end<=0:  # need to reverse it in mode "from the end of the array"
                self.end += self.n
            if end>0:
                self.end = end

            self.start = int(self.start)
            self.end = int(self.end)

            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

            self.values = np.zeros(self.M)

        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))
        self.avar = np.ones(self.M)
        self.best_avar = np.ones(self.M)
        self.target = self.IV[:, self.end]-self.IV[:, self.start]
        self.rate = 0.25  # convergence rate of the estimator


    def compute_best_avar(self):
        # Output : The asymptotic best avar
        a2true = self.noiseVar
        start, end = self.start, self.end

        self.best_avar = 8*np.sqrt(a2true*self.T_obs)*(self.R[:, end]-self.R[:, start])

    def compute_rho(self):
        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))

    def compute_kappa(self):
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))

    def info(self):
        print("Estimator :{0}".format(self.name))
        print("Student Bias :{0}".format(self.studentBias()))
        print("Student Std:{0}".format(self.studentStd()))
        print("Student RMSE:{0}".format(self.studentRMSE()))
        print("Empirical Loss:{0}".format(self.empLoss()))
        print("Theoretic Loss:{0}".format(self.theoLoss()))
        print("(Student Std)^2 x Theoretic Loss: {0}".format(self.studentStd()**2*self.theoLoss()))
        print("--------------------------")

    def toCsv(self, path):
        outputData = DataFrame()
        outputData["Target"] = self.target
        outputData["Values"] = self.values
        outputData["Global Rho"] = self.rho
        outputData["Global Kappa"] = self.kappa
        outputData["Avar"] = self.avar
        outputData["Best Avar"] = self.best_avar

        if path[-4:] == '.csv':
            outputData.to_csv(path[:-4]+self.name+"outputData.csv")
        else:
            outputData.to_csv(path+self.name+"outputData.csv")


class Tric_estimator(Estimator):
    __doc__ = """
        A Tric_estimator object is designed to perform efficient estimations of Integrated third power of volatility. It inherits the
        basic structure of a general Estimator object. The class Tric itself is abstract and is not intended to be used.

        %(mode)s

        %(param)s

        Use also : help() on Estimator class for more generic info
        """%{"mode": _class_init_mode_doc, "param": _class_param_doc}

    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape

            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            if end>=0:  # need to reverse it in mode "from the end of the array"
                self.end -= self.n

            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

        self.values = np.zeros_like(self.Y[:, 0])
        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))
        self.avar = np.ones_like(self.Y[:, 0])
        self.best_avar = np.ones_like(self.Y[:, 0])
        self.target = self.R[:, self.end]-self.R[:, self.start]
        self.rate = 0.25  # convergence rate of the estimator (To be checked for the IQ)


class Quart_estimator(Estimator):
    __doc__ = """
            A Quart_estimator object is designed to perform efficient estimations of Integrated fourth power of volatility. It inherits the
            basic structure of a general Estimator object. The class Quart_Estimator itself is abstract and is not intended to be used.

            %(mode)s

            %(param)s

            Use also : help() on Estimator class for more generic info
            """ % {"mode": _class_init_mode_doc, "param": _class_param_doc}

    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape

            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            if end>=0:  # need to reverse it in mode "from the end of the array"
                self.end -= self.n

            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

        self.values = np.zeros_like(self.Y[:, 0])
        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))
        self.avar = np.ones_like(self.Y[:, 0])
        self.best_avar = np.ones_like(self.Y[:, 0])
        self.target = self.Q[:, self.end]-self.Q[:, self.start]
        self.rate = 0.25  # convergence rate of the estimator (To be checked for the IQ)


class Rho_estimator(Estimator):
    __doc__ = """
            A Rho_estimator object is designed to perform efficient estimations of the heteroskedastic measure rho as defined in %(rho)s. It inherits the
            basic structure of a general Estimator object. The class Rho_Estimator itself is abstract and is not intended to be used.

            %(mode)s

            %(param)s

            Use also : help() on Estimator class for more generic info
            """ % {"mode": _class_init_mode_doc, "param": _class_param_doc, "rho":_ref_rho}

    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape

            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            if end>=0:  # need to reverse it in mode "from the end of the array"
                self.end -= self.n

            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

        self.values = np.zeros_like(self.Y[:, 0])
        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))
        self.avar = np.ones_like(self.Y[:, 0])
        self.best_avar = np.ones_like(self.Y[:, 0])
        self.target = self.rho
        self.rate = 0.25  # convergence rate of the estimator (To be checked for Rho)


class Kappa_estimator(Estimator):

    __doc__ = """
            A Kappa_estimator object is designed to perform efficient estimations of the heteroskedastic measure kappa as defined in %(kappa)s. It inherits the
            basic structure of a general Estimator object. The class Kappa_Estimator itself is abstract and is not intended to be used.

            %(mode)s

            %(param)s

            Use also : help() on Estimator class for more generic info
            """ % {"mode": _class_init_mode_doc, "param": _class_param_doc, "kappa":_ref_kappa}


    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape

            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            if end>=0:  # need to reverse it in mode "from the end of the array"
                self.end -= self.n

            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

        self.values = np.zeros_like(self.Y[:, 0])
        self.rho = (self.IV[:, self.end]-self.IV[:, self.start])/np.sqrt(
            self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))
        self.kappa = (self.R[:, self.end]-self.R[:, self.start])/(
            self.T_obs**0.25*((self.Q[:, self.end]-self.Q[:, self.start])**0.75))
        self.avar = np.ones_like(self.Y[:, 0])
        self.best_avar = np.ones_like(self.Y[:, 0])
        self.target = self.kappa
        self.rate = 0.25  # convergence rate of the estimator (To be checked for Kappa)


class Noise_estimator(Estimator):
    __doc__ = """
               A Noise_estimator object is designed to perform efficient estimations of the variance of the noise process. It inherits the
               basic structure of a general Estimator object. The class Noise_Estimator itself is abstract and is not intended to be used.

               %(mode)s

               %(param)s

               Use also : help() on Estimator class for more generic info
               """ % {"mode": _class_init_mode_doc, "param": _class_param_doc, "kappa": _ref_kappa}

    def __init__(self, start, end, modearg, name="myVolatilityEstimator", mode="fromDict"):

        super().__init__(start, end, modearg, name, mode)

        if mode == "fromDict":

            if "Y" in modearg:
                if len(modearg["Y"].shape) == 1:
                    self.Y = modearg["Y"].reshape((1, modearg["Y"].shape[0]))
                else:
                    self.Y = modearg["Y"]
            else:
                print("Bad argument : Y is missing")

            self.M, self.n = self.Y.shape

            if "IV" in modearg:
                if len(modearg["IV"].shape) == 1:
                    self.IV = modearg["IV"].reshape((1, modearg["IV"].shape[0]))
                else:
                    self.IV = modearg["IV"]
            else:
                self.IV = np.zeros_like(self.Y)

            if "Q" in modearg:
                if len(modearg["Q"].shape) == 1:
                    self.Q = modearg["Q"].reshape((1, modearg["Q"].shape[0]))
                else:
                    self.Q = modearg["Q"]
            else:
                self.Q = np.zeros_like(self.Y)

            if "R" in modearg:
                if len(modearg["R"].shape) == 1:
                    self.R = modearg["R"].reshape((1, modearg["R"].shape[0]))
                else:
                    self.R = modearg["R"]
            else:
                self.R = np.zeros_like(self.Y)

            if "noiseVar" in modearg:
                self.noiseVar = modearg["noiseVar"]
            else:
                self.noiseVar = 0

            self.d = modearg["d"]
            self.T = modearg["T"]

            self.delta = self.T/(self.n-1)

            self.end=end
            if end<0:  # need to reverse it in mode "from the end of the array"
                self.end += self.n


            self.T_obs = self.d*self.T*(1-(self.start-self.end-1)/self.n)
            self.n_obs = self.n-self.start+self.end+1

        self.values = np.zeros_like(self.Y[:, 0])
        self.avar = np.ones_like(self.Y[:, 0])
        self.best_avar = np.ones_like(self.Y[:, 0])
        self.target = self.noiseVar
        self.rate = 0.5  # convergence rate of the estimator

class Vol_QMLE(Vol_estimator):

    _class_name = "Vol_QMLE"
    _mother_class_name = "Vol_estimator"

    _setting_params = """
    Setting parameters for estimation :
    -----------------------------------

    %(cn)s needs no tuning parameter so that the method setParam() is useless here.
    """ % {"cn":_class_name}

    _estim = """
    Estimating :
    ------------

    Simply call the method estim() (no optional parameters) to fill the values field.
    """

    __doc__ = """
       A %(cn)s object is designed to perform estimations of Integrated Volatility by QMLE estimation as in %(qmle)s. It inherits the
       basic structure of a %(mcn)s object.

       %(mode)s

       %(param)s

       %(settingParams)s

       %(estim)s

       Use also : help() on %(mcn)s class for more generic info
       """ % {"cn":_class_name,"mcn":_mother_class_name,"mode": _class_init_mode_doc, "param": _class_param_doc, "qmle":_ref_qmle, "settingParams": _setting_params, "estim":_estim}

    def estim(self, infotype="none", returnLogLik=False, optionalArgs=()):
        end = np.zeros(self.M)
        if self.regular_obs == 1:
            end=self.end*np.ones(self.M)
        if self.regular_obs == 0:
            end = self.nbObs-1


        if infotype == "none":
            values = np.array([QMLE_manual(self.Y[iMC, self.start:int(end[iMC])], self.T_obs) for iMC in range(self.M)]) #use QMLE_package to go faster

        if infotype == "roll":
            #if returnLogLik:
            values = np.array([QMLE_manual_roll(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.T_obs,returnLogLik) for iMC in range(self.M)])
            #else:
            #    values = np.array([QMLE_manual_roll(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.T_obs) for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue=np.zeros(self.M)

        if infotype == "glostenharris":
            values = np.array([QMLE_manual_glostenharris(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.volume[iMC, self.start:int(end[iMC])], self.T_obs)[0:4] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)

        if infotype == "glostenharrisNoRoll":
            values = np.array([QMLE_manual_glostenharrisNoRoll(self.Y[iMC, self.start:end[iMC]],
                                                             self.info[iMC, self.start:end[iMC]],
                                                             self.volume[iMC, self.start:end[iMC]], self.T_obs)[0:3] for
                                   iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1 = np.zeros(self.M)

        if infotype == "glostenharrists":
            values = np.array([QMLE_manual_glostenharrists(self.Y[iMC, self.start:end[iMC]], self.info[iMC, self.start:end[iMC]], self.volume[iMC, self.start:end[iMC]], self.ts[iMC, self.start:end[iMC]], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)
            #self.knownNoise_estimatedvalue3 = np.zeros(self.M)

        if infotype == "ts":
            values = np.array([QMLE_manual_ts(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.ts[iMC, self.start:int(end[iMC])], self.T_obs)[0:5] for iMC in range(self.M)])
            self.knownNoise_estimatedvalue1=np.zeros(self.M)

        if infotype == "spread":
            #if returnLogLik:
            values = np.array([QMLE_manual_spread(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs, returnLogLik) for iMC in range(self.M)])
            #else:
            #    values = np.array([QMLE_manual_spread(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs)[0:5] for iMC in range(self.M)])

            #self.knownNoise_estimatedvalue1=np.zeros(self.M)

        if infotype == "spreadLog":
            values = np.array([QMLE_manual_spreadLog(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)

        if infotype == "spreadRoll":
            values = np.array([QMLE_manual_spreadRoll(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs, returnLogLik) for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)

        if infotype == "spreadLag":
            values = np.array([QMLE_manual_spreadLag(self.Y[iMC, self.start:end[iMC]], self.info[iMC, self.start:end[iMC]], self.spread[iMC, self.start:end[iMC]], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)

        if infotype == "volLim":
            values = np.array([QMLE_manual_volLim(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.volLim[iMC, self.start:int(end[iMC])], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)

        #if infotype == "volLim2":
        #    values = np.array([QMLE_manual_spreadRollVolTsVollim(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.volume[iMC, self.start:int(end[iMC])], self.ts[iMC, self.start:int(end[iMC])], self.volLim[iMC, self.start:int(end[iMC])], self.T_obs)[0:5] for iMC in range(self.M)])
        #    self.knownNoise_estimatedvalue1=np.zeros(self.M)

        if infotype == "OPI":
            values = np.array([QMLE_manual_OPI(self.Y[iMC, self.start:end[iMC]], self.OPI[iMC, self.start:end[iMC]], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)

        if infotype == "spreadVolLim":
            values = np.array([QMLE_manual_spreadVolLim(self.Y[iMC, self.start:end[iMC]], self.info[iMC, self.start:end[iMC]], self.spread[iMC, self.start:end[iMC]], self.volLim[iMC, self.start:end[iMC]], self.T_obs)[0:5] for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)

        if infotype == "spreadRollVolTsVollim":
            values = np.array([QMLE_manual_spreadRollVolTsVollim(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.volume[iMC, self.start:int(end[iMC])], self.ts[iMC, self.start:int(end[iMC])], self.volLim[iMC, self.start:int(end[iMC])], self.T_obs,returnLogLik) for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)
            #self.knownNoise_estimatedvalue3 = np.zeros(self.M)
            #self.knownNoise_estimatedvalue4 = np.zeros(self.M)
            #self.knownNoise_estimatedvalue5 = np.zeros(self.M)

        if infotype == "spreadRollTs":
            values = np.array([QMLE_manual_spreadRollTs(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.ts[iMC, self.start:int(end[iMC])], self.T_obs,returnLogLik) for iMC in range(self.M)])
            #self.knownNoise_estimatedvalue1=np.zeros(self.M)
            #self.knownNoise_estimatedvalue2 = np.zeros(self.M)
            #self.knownNoise_estimatedvalue3 = np.zeros(self.M)

        if self.mode == "fromDict":
            self.noise_estimator = Noise_QMLE(self.start, self.end, self.__dict__, self.name+" Noise", self.mode)

        #for iMC in range(self.M):
        self.values = values[:, 0]
        self.noise_estimator.values = values[:, 1]
        if infotype == "roll":
            self.knownNoise_estimatedvalue =values[:, 2]
        if infotype == "glostenharris":
            self.knownNoise_estimatedvalue1  = values[:, 2]
            self.knownNoise_estimatedvalue2  = values[:, 3]
        if infotype == "glostenharrisNoRoll":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "glostenharrists":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
            self.knownNoise_estimatedvalue3 = values[:, 4]
        if infotype == "ts":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "spread":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "spreadLog":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "spreadRoll":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
        if infotype == "spreadLag":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
        if infotype == "volLim":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "OPI":
            self.knownNoise_estimatedvalue1 = values[:, 2]
        if infotype == "spreadVolLim":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
        if infotype == "spreadRollVolTsVollim":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
            self.knownNoise_estimatedvalue3 = values[:, 4]
            self.knownNoise_estimatedvalue4 = values[:, 5]
            self.knownNoise_estimatedvalue5 = values[:, 6]
        if infotype == "spreadRollTs":
            self.knownNoise_estimatedvalue1 = values[:, 2]
            self.knownNoise_estimatedvalue2 = values[:, 3]
            self.knownNoise_estimatedvalue3  = values[:, 4]

        ###log likelihood value
        if returnLogLik:
            self.loglik = values[:,-1]

    def compute_avar(self):

        a2true = self.noiseVar
        start, end = [self.start, self.end]

        self.avar = 5*np.sqrt(a2true)*(self.Q[:, end]-self.Q[:, start])*self.T_obs/np.sqrt(
            self.IV[:, end]-self.IV[:, start])+3*np.sqrt(a2true)*(self.IV[:, end]-self.IV[:, start])**1.5

#    def estimate_avar(self, infotype="none"):
#        start, end = [self.startf, self.end]
#        self.estimate_avar_vol = np.zeros(self.M)
#        self.estimate_avar_vol = np.zeros(self.M)
#        a2true = self.noise_estimator.Values
#        self.avar = 5*np.sqrt(a2true)*(self.Q[:, end]-self.Q[:, start])*self.T_obs/np.sqrt(
#            self.IV[:, end]-self.IV[:, start])+3*np.sqrt(a2true)*(self.IV[:, end]-self.IV[:, start])**1.5

class Vol_Yingying(Vol_estimator):

    _class_name = "Vol_Yingying"
    _mother_class_name = "Vol_estimator"

    def estim(self, infotype="roll", optionalArgs=()):
        end = np.zeros(self.M)
        if self.regular_obs == 1:
            end = self.end * np.ones(self.M)
        if self.regular_obs == 0:
            end = self.nbObs - 1
        if infotype == "roll":
            valuesKnownNoise = np.array([knownNoise_Yingying_roll(self.Y[iMC, self.start:int(end[iMC])],
                                                    self.info[iMC, self.start:int(end[iMC])], self.T_obs) for iMC in
                                   range(self.M)])
            valuesVol = np.array([volatility_Yingying_roll(self.Y[iMC, self.start:int(end[iMC])],
                                                           self.info[iMC, self.start:int(end[iMC])], self.T_obs,
                                                           valuesKnownNoise[iMC]) for iMC in
                                  range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(self.M)
            self.values = np.zeros(self.M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]
                self.values[iMC] = valuesVol[iMC]
        if infotype == "spread":
            valuesKnownNoise = np.array([knownNoise_Yingying_spread(self.Y[iMC, self.start:int(end[iMC])],
                                                    self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs) for iMC in
                                   range(self.M)])
            valuesVol = np.array([volatility_Yingying_spread(self.Y[iMC, self.start:int(end[iMC])], self.info[iMC, self.start:int(end[iMC])], self.spread[iMC, self.start:int(end[iMC])], self.T_obs,valuesKnownNoise[iMC]) for iMC in range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(self.M)
            #self.values = np.zeros(M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]
                self.values[iMC] = valuesVol[iMC]
        if infotype == "spreadCanonical":
            # en chantier
            values = np.array([knownNoise_Yingying_spreadCanonical(self.Y[iMC, self.start:self.end],
                                                    self.info[iMC, self.start:self.end], self.spread[iMC, self.start:self.end], self.T_obs) for iMC in
                                   range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(self.M)
            #self.values = np.zeros(M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]
                self.values[iMC] = valuesVol[iMC]

class Vol_Yingying_QMLE(Vol_estimator):

    _class_name = "Vol_Yingying_QMLE"
    _mother_class_name = "Vol_estimator"

    def estim(self, infotype="roll", optionalArgs=()):
        if infotype == "roll":
            valuesKnownNoise = np.array([knownNoise_Yingying_roll(self.Y[iMC, self.start:self.end], self.info[iMC, self.start:self.end], self.T_obs) for iMC in
                                         range(self.M)])
            valuesVolUnkwnownNoise = np.array([QMLE_manual(self.Y[iMC, self.start:self.end] - valuesKnownNoise[iMC] * self.info[iMC, self.start:self.end], self.T_obs)[0:2] for iMC in range(self.M)]) #np.array([QMLE_package(self.Y[iMC, self.start:self.end] - valuesKnownNoise[iMC] * self.info[iMC, self.start:self.end],self.T_obs)[0:2]) for iMC in range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(M)
            self.values = np.zeros(M)
            self.unknownNoise_estimatedvalue = np.zeros(M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]
                self.values[iMC] = valuesVolUnkwnownNoise[iMC,0]
                self.unknownNoise_estimatedvalue[iMC] = valuesVolUnkwnownNoise[iMC, 1]
        if infotype == "spread":
            valuesKnownNoise = np.array([knownNoise_Yingying_spread(self.Y[iMC, self.start:self.end], self.info[iMC, self.start:self.end], self.spread[iMC, self.start:self.end], self.T_obs) for iMC in
                                         range(self.M)])
            valuesVolUnkwnownNoise = np.array([QMLE_manual(self.Y[iMC, self.start:self.end] - valuesKnownNoise[iMC] * self.info[iMC, self.start:self.end] * self.spread[iMC, self.start:self.end], self.T_obs)[0:2] for iMC in range(self.M)]) #np.array([QMLE_package(self.Y[iMC, self.start:self.end] - valuesKnownNoise[iMC] * self.info[iMC, self.start:self.end],self.T_obs)[0:2]) for iMC in range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(M)
            self.values = np.zeros(M)
            self.unknownNoise_estimatedvalue = np.zeros(M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]
                self.values[iMC] = valuesVolUnkwnownNoise[iMC,0]
                self.unknownNoise_estimatedvalue[iMC] = valuesVolUnkwnownNoise[iMC, 1]


    def compute_avar(self):

        a2true = self.noiseVar
        start, end = [self.start, self.end]

        self.avar = 5*np.sqrt(a2true)*(self.Q[:, end]-self.Q[:, start])*self.T_obs/np.sqrt(
            self.IV[:, end]-self.IV[:, start])+3*np.sqrt(a2true)*(self.IV[:, end]-self.IV[:, start])**1.5

class Vol_ChakerIV(Vol_estimator):

    _class_name = "Vol_chakerIV"
    _mother_class_name = "Vol_estimator"

    def estim(self, infotype="spread", optionalArgs=()):
        if infotype == "spread":
            valuesKnownNoise = np.array([estimationIV(self.Y[iMC, self.start:self.end],
                                                    self.info[iMC, self.start:self.end], self.spread[iMC, self.start:self.end], 1) for iMC in
                                   range(self.M)])
            self.knownNoise_estimatedvalue = np.zeros(self.M)
            for iMC in range(self.M):
                self.knownNoise_estimatedvalue[iMC] = valuesKnownNoise[iMC]


class Vol_RK(Vol_estimator):
    _class_name = "Vol_RK"
    _mother_class_name = "Vol_estimator"

    _setting_params = """
    Setting parameters for estimation :
    -----------------------------------

    For %(cn)s, the method setParam() requires a few information in the following order:
    - The name (string) of the kernel to be picked up from the keys of the dictionnary fun_dict_RK
    - the estimation mode (string) : -"oracle" for oracle autotuning of the tuning parameter H
                            -"manual" for a manual tuning of the tuning parameter H
                            -"adaptive" for the feasible autotuning
    - the class for preestimating the Quarticity (only if "adaptive")
    - the class for preestimating the Volatility (only if "adaptive")
    - the class for preestimating the Noise (only if "adaptive")

    """ % {"cn":_class_name}

    _estim = """
    Estimating :
    ------------

    Call the method estim() with the following optional arguments to fill the values field (in the following order):
    - if "oracle", no optional argument is needed.
    - if "manual", an array-like object of dimension M (number of trajectories) containing the size of H for each estimation.
    - if "adaptive", and pre-estimating classes for IV and IQ are Vol_PA, an array-like object of dimension M (number of trajectories) containing the tuning parameter theta for each pre_estimation.
            It is also possible to give no optional argument in that case and let the estimator decide the value of theta by itself.
    """

    __doc__ = """
       A %(cn)s object is designed to perform estimations of Integrated Volatility by realized kernel as in %(rk)s. It inherits the
       basic structure of a %(mcn)s object.

       %(mode)s

       %(param)s

       %(settingParams)s

       %(estim)s

       Use also : help() on %(mcn)s class for more generic info
       """ % {"cn":_class_name,"mcn":_mother_class_name,"mode": _class_init_mode_doc, "param": _class_param_doc, "rk":_ref_rk, "settingParams": _setting_params, "estim":_estim}


    def setParam(self, *param):
        #print(len(param))
        if param != ():
            kernel_type, self.estimMode = param[:2]

            if self.estimMode == "adaptive":
                self.preIQClass, self.preVolClass, self.preNoiseClass = param[2:]

            self.kernel_type = kernel_type
            self.k0, self.k1, self.k2 = fun_dict_RK[kernel_type][1:4]

            a2true = self.noiseVar
            self.xi2optim = a2true/(self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))**0.5
            self.Hoptim = c_star(self.kernel_type, self.rho)*np.sqrt(self.xi2optim)*np.sqrt(self.n_obs)
            self.c = c_star(self.kernel_type, self.rho)

            self.H = np.zeros(self.M)

    def estim(self, optionalArgs=()):

        if self.estimMode == "oracle":
            a2true = self.noiseVar
            rkArgs = np.array(
                [[a2true, self.Q[iMC, self.end]-self.Q[iMC, self.start], self.rho[iMC]] for iMC in range(self.M)])
            self.H = np.array([c_star(self.kernel_type, self.rho[iMC])*np.sqrt(
                a2true/np.sqrt(self.T_obs*(self.Q[iMC, self.end]-self.Q[iMC, self.start])))*np.sqrt(self.n_obs)
                               for iMC in range(self.M)])
            self.values = np.array([realized_kernel_border(self.Y[iMC, :], self.T_obs, self.kernel_type, rkArgs[iMC],
                                                           self.estimMode, self.start, self.end) for iMC in
                                    range(self.M)])

        elif self.estimMode == "manual":
            self.H = np.array(optionalArgs[0])
            self.values = np.array([realized_kernel_border(self.Y[iMC, :], self.T_obs, self.kernel_type,
                                                           optionalArgs[0][iMC], self.estimMode, self.start, self.end)
                                    for iMC in range(self.M)])

        elif self.estimMode == "adaptive":

            if self.mode == "fromDict":
                self.pre_estimatorRho = Rho_direct(self.start, self.end, self.__dict__, "", self.mode)

                if self.preIQClass == Quart_PA:
                    if optionalArgs == ():
                        theta_pre = np.sqrt(self.T_obs)*np.ones(self.M)
                    else:
                        theta_pre = optionalArgs[0]
                    self.pre_estimatorRho.setParam(self.preVolClass, self.preIQClass, (theta_pre,),
                                                   ("triang_ker", "manual", None, None), (theta_pre,),
                                                   ("triang_ker", "manual"))
                else:
                    self.pre_estimatorRho.setParam(self.preVolClass, self.preIQClass)
                self.pre_estimatorRho.estim()

                if self.preVolClass == Vol_QMLE:  # particular case
                    # The noise is automatically computed in the QMLE routine.
                    self.pre_estimatorNoise = self.pre_estimatorRho.IVest.noise_estimator
                else:
                    self.pre_estimatorNoise = self.preNoiseClass(self.start, self.end, self.__dict__, "", self.mode)
                    self.pre_estimatorNoise.estim()

                rho_global_est = self.pre_estimatorRho.values
                Q_est = self.pre_estimatorRho.IQest.values
                a2true_est = self.pre_estimatorNoise.values

                rkArgs = np.array([[a2true_est[iMC], Q_est[iMC], rho_global_est[iMC]] for iMC in range(self.M)])
                self.H = np.array([c_star(self.kernel_type, rho_global_est[iMC])*np.sqrt(
                    a2true_est[iMC]/np.sqrt(self.T_obs*(Q_est[iMC])))*np.sqrt(self.n_obs) for iMC in
                                   range(self.M)])
                self.values = np.array([realized_kernel_border(self.Y[iMC, :], self.T_obs, self.kernel_type,
                                                               rkArgs[iMC], "oracle", self.start, self.end) for iMC in
                                        range(self.M)])

    def compute_avar(self):
        # Output : The asymptotic theoretic avar of the QMLE as in RK paper
        a2true = self.noiseVar
        start, end = [self.start, self.end]
        d_rk = self.k0*self.k2/(self.k1**2)
        self.avar = np.sqrt(a2true)*(self.T_obs*(self.Q[:, end]-self.Q[:, start]))**0.75*(16/3)*np.sqrt(
            self.rho*self.k0*self.k1)*((1/np.sqrt(1+np.sqrt(1+3*d_rk/self.rho**2)))+np.sqrt(
            1+np.sqrt(1+3*d_rk/self.rho**2)))

    def compute_avarH(self):
        a2true = self.noiseVar
        start, end = [self.start, self.end]
        rho_global = self.rho
        c = self.H/np.sqrt(self.xi2optim*self.n_obs)
        self.avarH = 4*np.sqrt(a2true)*(self.T_obs*(self.Q[:, end]-self.Q[:, start]))**0.75*(
            c*self.k0+2/c*rho_global*self.k1+self.k2/c**3)

    def g_prime_rk(self, c):
        return c**4*self.k0-2*c**2*self.k1*self.rho-3*self.k2

    def g_rk(self, rho):
        d_rk = self.k0*self.k2/(self.k1**2)
        return (16/3)*np.sqrt(rho*self.k0*self.k1)*(
            (1/np.sqrt(1+np.sqrt(1+3*d_rk/rho**2)))+np.sqrt(1+np.sqrt(1+3*d_rk/rho**2)))

    def info(self):
        print("Estimator :{0}".format(self.name))
        print("Student Bias :{0}".format(self.studentBias()))
        print("Student Std:{0}".format(self.studentStd()))
        print("Student RMSE:{0}".format(self.studentRMSE()))
        print("Empirical Loss:{0}".format(self.empLoss()))
        print("Theoretic Loss:{0}".format(self.theoLoss()))
        print("(Student Std)^2 x Theoretic Loss: {0}".format(self.studentStd()**2*self.theoLoss()))
        print("loss due to H in avar:{0}".format(np.mean(self.avarH/self.avar)))
        print("--------------------------")

    def toCsv(self, path):
        outputData = DataFrame()
        outputData["Target"] = self.target
        outputData["Values"] = self.values
        outputData["Global Rho"] = self.rho
        outputData["Global Kappa"] = self.kappa
        outputData["Avar"] = self.avar
        outputData["Best Avar"] = self.best_avar
        self.compute_avarH()
        outputData["AvarH Ratio"] = self.avarH/self.avar
        outputData["H"] = self.H
        outputData["H star"] = self.Hoptim
        outputData["xi2 star"] = self.xi2optim

        if path[-4:] == '.csv':
            outputData.to_csv(path[:-4]+self.name+"outputData.csv")
        else:
            outputData.to_csv(path+self.name+"outputData.csv")

class Vol_RK_depnoise(Vol_estimator):
    _class_name = "Vol_RK_depnoise"
    _mother_class_name = "Vol_estimator"

    _setting_params = """
    Setting parameters for estimation :
    -----------------------------------

    For %(cn)s, the method setParam() requires a few information in the following order:
    - The name (string) of the kernel to be picked up from the keys of the dictionnary fun_dict_RK
    - the estimation mode (string) : -"oracle" for oracle autotuning of the tuning parameter H
                            -"manual" for a manual tuning of the tuning parameter H
                            -"adaptive" for the feasible autotuning
    - the class for preestimating the Quarticity (only if "adaptive")
    - the class for preestimating the Volatility (only if "adaptive")
    - the class for preestimating the Noise (only if "adaptive")

    """ % {"cn":_class_name}

    _estim = """
    Estimating :
    ------------

    Call the method estim() with the following optional arguments to fill the values field (in the following order):
    - if "oracle", no optional argument is needed.
    - if "manual", an array-like object of dimension M (number of trajectories) containing the size of H for each estimation.
    - if "adaptive", and pre-estimating classes for IV and IQ are Vol_PA, an array-like object of dimension M (number of trajectories) containing the tuning parameter theta for each pre_estimation.
            It is also possible to give no optional argument in that case and let the estimator decide the value of theta by itself.
    """

    __doc__ = """
       A %(cn)s object is designed to perform estimations of Integrated Volatility by realized kernel as in %(rk)s. It inherits the
       basic structure of a %(mcn)s object.

       %(mode)s

       %(param)s

       %(settingParams)s

       %(estim)s

       Use also : help() on %(mcn)s class for more generic info
       """ % {"cn":_class_name,"mcn":_mother_class_name,"mode": _class_init_mode_doc, "param": _class_param_doc, "rk":_ref_rk, "settingParams": _setting_params, "estim":_estim}


    def setParam(self, *param):
        #print(len(param))
        if param != ():
            kernel_type, self.estimMode = param[:2]

            if self.estimMode == "adaptive":
                self.preIQClass, self.preVolClass, self.preNoiseClass = param[2:]

            self.kernel_type = kernel_type
            self.k0, self.k1, self.k2 = fun_dict_RK[kernel_type][1:4]

            a2true = self.noiseVar
            self.xi2optim = a2true/(self.T_obs*(self.Q[:, self.end]-self.Q[:, self.start]))**0.5
            self.Hoptim = c_star(self.kernel_type, self.rho)*np.sqrt(self.xi2optim)*np.sqrt(self.n_obs)
            self.c = c_star(self.kernel_type, self.rho)

            self.H = np.zeros(self.M)

    def estim(self, optionalArgs=()):

        if self.estimMode == "oracle":
            pass

        elif self.estimMode == "manual":
            self.H, self.gamma = np.array(optionalArgs[0]), optionalArgs[1]
            self.values = np.array([realized_kernel_border_depnoise(self.Y[iMC, :], self.T_obs, self.kernel_type,
                                                           optionalArgs[0][iMC],self.gamma, self.estimMode, self.start, self.end)
                                    for iMC in range(self.M)])

        elif self.estimMode == "adaptive":

            if self.mode == "fromDict":
                self.pre_estimatorRho = Rho_direct(self.start, self.end, self.__dict__, "", self.mode)

                if self.preIQClass == Quart_PA:
                    #if optionalArgs == ():
                    #else:
                    #    theta_pre = optionalArgs[0]
                    theta_pre = np.sqrt(self.T_obs) * np.ones(self.M)
                    self.gamma = optionalArgs[0]
                    self.pre_estimatorRho.setParam(self.preVolClass, self.preIQClass, (theta_pre,),
                                                   ("triang_ker", "manual", None, None), (theta_pre,),
                                                   ("triang_ker", "manual"))
                else:
                    self.pre_estimatorRho.setParam(self.preVolClass, self.preIQClass)
                self.pre_estimatorRho.estim()

                if self.preVolClass == Vol_QMLE:  # particular case
                    # The noise is automatically computed in the QMLE routine.
                    self.pre_estimatorNoise = self.pre_estimatorRho.IVest.noise_estimator
                else:
                    self.pre_estimatorNoise = self.preNoiseClass(self.start, self.end, self.__dict__, "", self.mode)
                    self.pre_estimatorNoise.estim()

                rho_global_est = self.pre_estimatorRho.values
                Q_est = self.pre_estimatorRho.IQest.values
                a2true_est = self.pre_estimatorNoise.values

                rkArgs = np.array([[a2true_est[iMC], Q_est[iMC], rho_global_est[iMC], self.gamma] for iMC in range(self.M)])
                self.H = np.array([c_star(self.kernel_type, rho_global_est[iMC])*np.sqrt(
                    a2true_est[iMC]/np.sqrt(self.T_obs*(Q_est[iMC])))*np.sqrt(self.n_obs) for iMC in
                                   range(self.M)])
                self.values = np.array([realized_kernel_border_depnoise(self.Y[iMC, :], self.T_obs, self.kernel_type,
                                                               rkArgs[iMC], "oracle", self.start, self.end) for iMC in
                                        range(self.M)])

    def compute_avar(self):
        pass

    def compute_avarH(self):
        pass

    def g_prime_rk(self, c):
        return c**4*self.k0-2*c**2*self.k1*self.rho-3*self.k2

    def g_rk(self, rho):
        d_rk = self.k0*self.k2/(self.k1**2)
        return (16/3)*np.sqrt(rho*self.k0*self.k1)*(
            (1/np.sqrt(1+np.sqrt(1+3*d_rk/rho**2)))+np.sqrt(1+np.sqrt(1+3*d_rk/rho**2)))

    def info(self):
        print("Estimator :{0}".format(self.name))
        print("Student Bias :{0}".format(self.studentBias()))
        print("Student Std:{0}".format(self.studentStd()))
        print("Student RMSE:{0}".format(self.studentRMSE()))
        print("Empirical Loss:{0}".format(self.empLoss()))
        print("Theoretic Loss:{0}".format(self.theoLoss()))
        print("(Student Std)^2 x Theoretic Loss: {0}".format(self.studentStd()**2*self.theoLoss()))
        print("loss due to H in avar:{0}".format(np.mean(self.avarH/self.avar)))
        print("--------------------------")

    def toCsv(self, path):
        outputData = DataFrame()
        outputData["Target"] = self.target
        outputData["Values"] = self.values
        outputData["Global Rho"] = self.rho
        outputData["Global Kappa"] = self.kappa
        outputData["Avar"] = self.avar
        outputData["Best Avar"] = self.best_avar
        self.compute_avarH()
        outputData["AvarH Ratio"] = self.avarH/self.avar
        outputData["H"] = self.H
        outputData["H star"] = self.Hoptim
        outputData["xi2 star"] = self.xi2optim

        if path[-4:] == '.csv':
            outputData.to_csv(path[:-4]+self.name+"outputData.csv")
        else:
            outputData.to_csv(path+self.name+"outputData.csv")

class Vol_PA(Vol_estimator):
    _class_name = "Vol_PA"
    _mother_class_name = "Vol_estimator"

    _setting_params = """
    Setting parameters for estimation :
    -----------------------------------

    For %(cn)s, the method setParam() requires a few information in the following order:
    - The name (string) of the weight function to be picked up from the keys of the dictionnary fun_dict_PA
    - the estimation mode (string) : -"oracle" for oracle autotuning of the tuning parameter theta
                            -"manual" for a manual tuning of the tuning parameter theta
                            -"adaptive" for the feasible autotuning
    - the class for preestimating the Volatility (only if "adaptive")
    - the class for preestimating the Noise (only if "adaptive")

    """ % {"cn":_class_name}

    _estim = """
    Estimating :
    ------------

    Call the method estim() with the following optional arguments to fill the values field (in the following order):
    - if "oracle", no optional argument is needed.
    - if "manual", an array-like object of dimension M (number of trajectories) containing the size of theta for each estimation.
    - if "adaptive", and pre-estimating classes for IV is Vol_PA, an array-like object of dimension M (number of trajectories) containing the tuning parameter theta for each pre_estimation.
            It is also possible to give no optional argument in that case and let the estimator decide the value of theta by itself.
    """

    __doc__ = """
       A %(cn)s object is designed to perform estimations of Integrated Volatility by pre-averaging method as in %(pae)s. It inherits the
       basic structure of a %(mcn)s object.

       %(mode)s

       %(param)s

       %(settingParams)s

       %(estim)s

       Use also : help() on %(mcn)s class for more generic info
       """ % {"cn":_class_name,"mcn":_mother_class_name,"mode": _class_init_mode_doc, "param": _class_param_doc, "pae":_ref_pae, "settingParams": _setting_params, "estim":_estim}


    def setParam(self, *param):
        if param != ():
            self.known_fun, self.estimMode, self.preVolClass, self.preNoiseClass = param

        else:
            self.known_fun = "triang_ker"

    def estim(self, optionalArgs=()):

        g = fun_dict_PA[self.known_fun][0]

        if self.estimMode == "manual" or self.estimMode == "default":
            if self.estimMode == "manual":
                theta = optionalArgs[0]
            else:
                theta = np.sqrt(self.T_obs)*np.ones(self.M)  # default value

            k = theta/np.sqrt(self.delta)
            self.k = k
            self.values = np.array(
                [C_hat(self.Y[iMC, self.start:self.end], self.delta, int(k[iMC]), g, self.T_obs, self.known_fun) for iMC
                 in range(self.M)])


        else:

            a = 0.5
            w = 5/6
            l = min(self.T_obs/self.delta**w, self.T_obs/self.delta)
            l_prime = int(a*l)
            L = int(self.T_obs/(l*self.delta))
            self.L = L

            c = np.zeros((self.M, L))
            gamma = np.zeros((self.M, L))

            #####PRE ESTIMATION PHASE
            if self.estimMode == "adaptive":

                if self.mode == "fromDict":

                    if self.preVolClass == Vol_QMLE:  # particular case
                        self.pre_estimatorVol = [self.preVolClass(self.start+r*l-l_prime,
                                                                           self.start+r*l, self.__dict__, "",
                                                                           self.mode) for r in range(1, L)]

                        for r in range(L-1):
                            self.pre_estimatorVol[r].estim()  # The noise is automatically computed in the QMLE routine.

                        self.pre_estimatorNoise = [self.pre_estimatorVol[r].noise_estimator for r in range(L-1)]

                    if self.preVolClass == Vol_PA:
                        if optionalArgs == ():
                            theta_pre = np.maximum(np.minimum(np.sqrt(self.T_obs/self.delta), (1-a)*l), 2.)*np.sqrt(
                                self.delta)*np.ones(self.M)
                        else :
                            theta_pre = optionalArgs[0]
                        k_pre = theta_pre[0]/np.sqrt(self.delta)
                        self.pre_estimatorVol = [self.preVolClass(self.start+r*l-l_prime-int(k_pre),
                                                                           self.start+r*l-int(k_pre),
                                                                           self.__dict__, "", self.mode) for r in
                                                          range(1, L)]
                        self.pre_estimatorNoise = [self.preNoiseClass(self.start+r*l-l_prime,
                                                                               self.start+r*l, self.__dict__, "",
                                                                               self.mode) for r in range(1, L)]

                        for r in range(L-1):
                            self.pre_estimatorVol[r].setParam(self.known_fun, "manual", None,None)
                            self.pre_estimatorVol[r].estim((theta_pre,))

                            self.pre_estimatorNoise[r].estim()

                    else:  # May crash !
                        self.pre_estimatorVol = [self.preVolClass(self.start+r*l-l_prime,
                                                                           self.start+r*l, self.__dict__, "",
                                                                           self.mode) for r in range(1, L)]
                        self.pre_estimatorNoise = [self.preNoiseClass(self.start+r*l-l_prime,
                                                                               self.start+r*l, self.__dict__, "",
                                                                               self.mode) for r in range(1, L)]

                for r in range(1, L):
                    c[:, r] = self.pre_estimatorVol[r-1].values/(l_prime*self.delta)
                    gamma[:, r] = self.pre_estimatorNoise[r-1].values

            elif self.estimMode == "oracle":

                for iMC in range(self.M):
                    c[iMC, 1:L] = 1/(l_prime*self.delta)*np.array(
                        [self.IV[iMC, self.start+r*l]-self.IV[iMC, self.start+r*l-l_prime] for r in
                         range(1, L)])

                gamma = self.noiseVar*np.ones((self.M, L))

            ######END OF PREESTIMATION PHASE
            psi_ggprime = Psi(g, g, self.known_fun)

            theta = np.zeros((self.M, L))
            theta[c == 0] = np.sqrt(self.T_obs)
            theta[c>0] = np.sqrt(gamma[c>0]/(psi_ggprime*c[c>0]))

            self.c = c
            self.gamma = gamma
            self.k = np.zeros((self.M, L))
            for iMC in range(self.M):
                self.values[iMC], self.k[iMC, :] = preaverage(self.Y[iMC, self.start:self.end], self.delta, g, g,
                                                              self.T_obs, a, w, self.known_fun, "none", theta[iMC, :])

    def compute_avar(self):
        a2true = self.noiseVar
        start, end = [self.start, self.end]
        self.avar = alpha_avar(fun_dict_PA[self.known_fun][0], fun_dict_PA[self.known_fun][0],
                               self.known_fun)*8*np.sqrt(a2true*self.T_obs)*(self.R[:, end]-self.R[:, start])

    def toCsv(self, path):
        outputData = DataFrame()
        outputData["Target"] = self.target
        outputData["Values"] = self.values
        outputData["Global Rho"] = self.rho
        outputData["Global Kappa"] = self.kappa
        outputData["Avar"] = self.avar
        outputData["Best Avar"] = self.best_avar
        outputData["k_mean"] = self.k.mean(axis=1)
        outputData["L"] = self.L*np.ones(self.M)

        if path[-4:] == '.csv':
            outputData.to_csv(path[:-4]+self.name+"outputData.csv")
        else:
            outputData.to_csv(path+self.name+"outputData.csv")


class Vol_TSRV(Vol_estimator):
    _class_name = "Vol_TSRV"
    _mother_class_name = "Vol_estimator"

    _setting_params = """
        Setting parameters for estimation :
        -----------------------------------

        For %(cn)s, the method setParam() is useless.

        """ % {"cn": _class_name}

    _estim = """
        Estimating :
        ------------

        Call the method estim() with the optional arguments :
        -K
        -J
        """

    __doc__ = """
           A %(cn)s object is designed to perform estimations of Integrated Volatility by TSRV method as in %(tsrv)s. It inherits the
           basic structure of a %(mcn)s object.

           %(mode)s

           %(param)s

           %(settingParams)s

           %(estim)s

           Use also : help() on %(mcn)s class for more generic info
           """ % {"cn": _class_name, "mcn": _mother_class_name, "mode": _class_init_mode_doc, "param": _class_param_doc,
                  "tsrv": _ref_tsrv, "settingParams": _setting_params, "estim": _estim}

    def estim(self, optionalArgs=()):
        K, J = optionalArgs
        self.values = np.array([TSRV(self.Y[iMC, self.start:self.end], K, J) for iMC in range(self.M)])


class Vol_Local(Vol_estimator):  # Local version of the mother class "estClass" in setParam()
    __doc__ = """
    Vol_Local is now obsolete. Use the function localize on Vol_Estimator instead !
    """

    def setParam(self, *param):
        b, estClass = param[:2]
        paramMother = param[2:]

        self.nb_blocks = b
        self.estClass = estClass

        if self.mode == "fromDict":
            self.sub_estimator = [estClass(self.start+int(i/self.nb_blocks*self.n_obs),
                                                    min(self.start+int((i+1)/self.nb_blocks*self.n_obs),
                                                        self.n-1), self.__dict__, self.name+str(i), self.mode) for i
                                           in range(self.nb_blocks)]

        for est in self.sub_estimator:
            est.setParam(*paramMother)

    def estim(self, optionalArgs=()):

        for est in self.sub_estimator:
            est.estim(optionalArgs)

        self.values = sum([est.values for est in self.sub_estimator])

    def compute_avar(self):

        for est in self.sub_estimator:
            est.compute_avar()

        self.avar = self.nb_blocks**(self.rate*2)*sum(
            [est.avar for est in self.sub_estimator])  # first term necessary for scaling


class Quart_PA(Quart_estimator):
    _class_name = "Quart_PA"
    _mother_class_name = "Quart_estimator"

    _setting_params = """
       Setting parameters for estimation :
       -----------------------------------

       For %(cn)s, the method setParam() requires a few information in the following order:
       - The name (string) of the weight function to be picked up from the keys of the dictionnary fun_dict_PA
       - the estimation mode (string) : -"manual" for a manual tuning of the tuning parameter theta

       """ % {"cn": _class_name}

    _estim = """
       Estimating :
       ------------

       Call the method estim() with the following optional arguments to fill the values field (in the following order):
       - if "manual", an array-like object of dimension M (number of trajectories) containing the size of theta for each estimation.
       """

    __doc__ = """
          A %(cn)s object is designed to perform estimations of Integrated Quarticity by pre-averaging method as in %(pae)s. It inherits the
          basic structure of a %(mcn)s object.

          %(mode)s

          %(param)s

          %(settingParams)s

          %(estim)s

          Use also : help() on %(mcn)s class for more generic info
          """ % {"cn": _class_name, "mcn": _mother_class_name, "mode": _class_init_mode_doc, "param": _class_param_doc,
                 "pae": _ref_pae, "settingParams": _setting_params, "estim": _estim}

    def setParam(self, *param):
        if param != ():
            self.known_fun, self.estimMode = param
        else:
            self.known_fun = "triang_ker"

    def estim(self, optionalArgs=()):

        g = fun_dict_PA[self.known_fun][0]
        k = np.zeros(self.M)

        if self.estimMode == "manual" or self.estimMode == "default":
            if self.estimMode == "manual":
                theta = optionalArgs[0]
            else:
                theta = np.sqrt(self.T_obs)*np.ones(self.M)  # default value
            k = theta/np.sqrt(self.delta)

        self.values = np.array(
            [Q_hat(self.Y[iMC, self.start:self.end], self.delta, int(k[iMC]), g, self.T_obs, self.known_fun) for iMC in
             range(self.M)])
        self.k = k

class Noise_Realized(Noise_estimator):

    _class_name = "Noise_Realized"
    _mother_class_name = "Noise_estimator"

    _setting_params = """
           Setting parameters for estimation :
           -----------------------------------

           For %(cn)s, the method setParam() is useless.

           """ % {"cn": _class_name}

    _estim = """
           Estimating :
           ------------

           Call the method estim() with no optional arguments.
           """

    __doc__ = """
              A %(cn)s object is designed to perform estimations of Noise Variance by the classical method of realized variance. It inherits the
              basic structure of a %(mcn)s object.

              %(mode)s

              %(param)s

              %(settingParams)s

              %(estim)s

              Use also : help() on %(mcn)s class for more generic info
              """ % {"cn": _class_name, "mcn": _mother_class_name, "mode": _class_init_mode_doc,
                     "param": _class_param_doc,
                     "tsrv": _ref_tsrv, "settingParams": _setting_params, "estim": _estim}

    def estim(self, optionalArgs=()):
        self.values = 1/(2*self.n_obs)*np.array(
            [sum(np.diff(self.Y[iMC, self.start:self.end])**2) for iMC in range(self.M)])


class Noise_QMLE(Noise_estimator):


    _class_name = "Noise_QMLE"
    _mother_class_name = "Noise_estimator"

    _setting_params = """
           Setting parameters for estimation :
           -----------------------------------

           For %(cn)s, the method setParam() is useless.

           """ % {"cn": _class_name}

    _estim = """
           Estimating :
           ------------

           Call the method estim() with no optional arguments.
           """

    __doc__ = """
              A %(cn)s object is designed to perform estimations of Noise Variance by QMLE estimation as in %(qmle)s.. It inherits the
              basic structure of a %(mcn)s object.

              %(mode)s

              %(param)s

              %(settingParams)s

              %(estim)s

              Use also : help() on %(mcn)s class for more generic info
              """ % {"cn": _class_name, "mcn": _mother_class_name, "mode": _class_init_mode_doc,
                     "param": _class_param_doc,
                     "qmle": _ref_qmle, "settingParams": _setting_params, "estim": _estim}


    def estim(self, optionalArgs=()):
        self.values = np.array([QMLE_manual(self.Y[iMC, self.start:self.end], self.T_obs)[1] for iMC in range(self.M)])

class Rho_direct(Rho_estimator):


    _class_name = "Rho_direct"
    _mother_class_name = "Rho_estimator"

    _setting_params = """
           Setting parameters for estimation :
           -----------------------------------

           For %(cn)s, the method setParam() requires a few arguments in the following order:
           -IVclass, the class for estimating the integrated volatility
           -IQclass, class for estimating the integrated quarticity
           -parameters (tuple) to feed the setParam method of IVclass.
           -parameters (tuple) to feed the setParam method of IQclass.
           -parameters (tuple) to feed the estim method of IVclass.
           -parameters (tuple) to feed the estim method of IQclass.

           """ % {"cn": _class_name}

    _estim = """
           Estimating :
           ------------

           Call the method estim() with no optional arguments.
           """

    __doc__ = """
              A %(cn)s object is designed to perform estimations of the heteroskedastic measure rho by direct estimation of the integrated volatility and the integrated quarticity as in %(rho)s.. It inherits the
              basic structure of a %(mcn)s object. Note that the estimation procedure does not restrain the estimates below the critical value 1 as
              it induces a large downward bias especially when rho is close to 1. This can be modified by uncommenting one line in
              the estim() method code. See below.

              %(mode)s

              %(param)s

              %(settingParams)s

              %(estim)s

              Use also : help() on %(mcn)s class for more generic info
              """ % {"cn": _class_name, "mcn": _mother_class_name, "mode": _class_init_mode_doc,
                     "param": _class_param_doc,
                     "rho": _ref_rho, "settingParams": _setting_params, "estim": _estim}


    def setParam(self, IVclass, IQclass, IVarg=(), IVparam=(), IQarg=(), IQparam=()):
        if self.mode == "fromDict":
            self.IVest = IVclass(self.start, self.end, self.__dict__, "IVest", self.mode)
            self.IQest = IQclass(self.start, self.end, self.__dict__, "IQest", self.mode)

        self.IVest.setParam(*IVparam)
        self.IQest.setParam(*IQparam)

        self.IVarg = IVarg
        self.IQarg = IQarg

    def estim(self, optionalArg=()):
        self.IVest.estim(self.IVarg)
        self.IQest.estim(self.IQarg)

        #uncomment the following line to force the estimates to stay below 1.
        #self.IQest.values[self.IQest.values<self.IVest.values**2/self.T_obs] = self.IVest.values[
        #                                                                           self.IQest.values<self.IVest.values**2/self.T_obs]**2/self.T_obs

        self.IQest.values[self.IQest.values < 0] = self.IVest.values[self.IQest.values < 0]**2/self.T_obs
        self.values = np.maximum(self.IVest.values/(np.sqrt(self.T_obs)*np.sqrt(self.IQest.values)),
                                 0.1)

    def avar_qmle(self):
        """

        :return: the avar_qmle that one would get if the IV and the IQ were the estimated values.
        """
        return avar_qmle(self.IVest.values,self.IQest.values,self.T_obs)

    def avar_rk(self, kernel = "th2"):
        """

        :return: the avar_rk that one would get if the IV and the IQ were the estimated values.
        """
        return avar_rk(self.IVest.values,self.IQest.values,self.T_obs, kernel)


class Kappa_direct(Kappa_estimator):

    def setParam(self, ITclass, IQclass, ITarg=(), ITparam=(), IQarg=(), IQparam=()):
        if self.mode == "fromDict":
            self.ITest = ITclass(self.start, self.end, self.__dict__, "ITest", self.mode)
            self.IQest = IQclass(self.start, self.end, self.__dict__, "IQest", self.mode)

        self.ITest.setParam(*ITparam)
        self.IQest.setParam(*IQparam)

        self.ITarg = ITarg
        self.IQarg = IQarg

    def estim(self, optionalArg=()):
        self.ITest.estim(self.ITarg)
        self.IQest.estim(self.IQarg)

        self.values = self.ITest.values/((self.T_obs**4)*(self.IQest.values)**0.75)

def localize(MotherClass):
    """
    Returns a local version of the argument MotherClass. (should be picked up from the abstract classes Vol_estimator, Quart_estimator,etc...)

    An instance of the local class can be created using the same specification as the one from MotherClass, along with
    two additional couples (key,value) in the argument dict :
    - "b" : number of blocks (int) for the local method
    - "estClass" : real class to be used for the estimation procedure. For instance, Vol_QMLE, Vol_RK .... if MotherClass is Vol_estimator

    Use help() on the local class itself for more info

    :param MotherClass:
    :return: a local class version of MotherClass
    """
    class Estimator_Local(MotherClass):  # Local version of the parameter class "estClass"

        __doc__ = """

        This is the local version of MotherClass following the methodology in %(clinetPotiron)s.
        An instance of the local class can be created using the same specification as the one from MotherClass, along with
        two additional couples (key,value) in the argument dict :
        - "b" : number of blocks (int) for the local method
        - "estClass" : real class to be used for the estimation procedure. For instance, Vol_QMLE, Vol_RK .... if MotherClass is Vol_estimator

        the new instance has an additional field "sub_estimator" which is a list of b estimators generated by estClass, and which
        represent the local estimators on each block of size T/b.


        setting param :
        ---------------

        Call the method setParam() with the parameters that are given to each of the sub estimators stocked in sub_estimator.
        This means that the same parameters are given to all the sub_estimators

        estimating :
        ------------

        Call the method estim() with the optional args that should be given to the sub estimators.
        This means that the same args are given to all the sub_estimators

        The field "values" is then filled by aggregation (sum) of the local estimates.

        """%{"clinetPotiron":_ref_efficient_asymptotic}

        def __init__(self, start, end, modearg, name="myEstimator", mode="fromDict"):

            MotherClass.__init__(self, start, end, modearg, name, mode)
            if self.mode == "fromDict":
                if "estClass" in modearg :
                    if issubclass(modearg["estClass"], MotherClass):
                        self.estClass = modearg["estClass"]
                    else:
                        raise ValueError("Error :  estClass should inherit from MotherClass")
                if "b" in modearg :
                    self.nb_blocks = modearg["b"]

                self.sub_estimator = [self.estClass(self.start+int(i/self.nb_blocks*self.n_obs),
                                                                 min(self.start+int((i+1)/self.nb_blocks*self.n_obs),
                                                                     self.n-1), self.__dict__, self.name+str(i),
                                                                 self.mode) for i in range(self.nb_blocks)]

            #TODO : Handle exceptions

        def setParam(self, *param):

            for est in self.sub_estimator:
                est.setParam(*param)

        def estim(self, infotype="none", optionalArgs=()):

            for est in self.sub_estimator:
                est.estim(infotype, optionalArgs)

            self.values = sum([est.values for est in self.sub_estimator])

        def compute_avar(self):

            for est in self.sub_estimator:
                est.compute_avar()

            self.avar = self.nb_blocks**(self.rate*2)*sum(
                [est.avar for est in self.sub_estimator])  # first term necessary for scaling

        def aggreg(self,funName):
            """

            :param funName:
            :return: aggregates (sums) over the sub estimators the result of their methods "funName"
            """

            if funName in self.estClass.__dict__:
                return sum([self.estClass.__dict__[funName](sub) for sub in self.sub_estimator])

        def aggreg_var(self,funName):

            """
            :param funName:
            :return: aggregates (sums) and scale by b**(rate*2) over the sub estimators the result of their methods "funName"
            This is useful　when aggregating AVARs.
            """
            if funName in self.estClass.__dict__:
                return self.nb_blocks**(self.rate*2)*sum([self.estClass.__dict__[funName](sub) for sub in self.sub_estimator])

    return Estimator_Local


